﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Text;

using Microsoft.Extensions.Logging;

using Newtonsoft.Json;
using Newtonsoft.Json.Converters;
using Newtonsoft.Json.Linq;
using ServiceApp.Enums;
using ServiceApp.Models;

namespace ServiceApp.Services
{
    public interface ISettingsService
    {
        SettingsModel Settings { get; set; }

        ServiceResult LoadSettings();
        ServiceResult SaveSettings();
        ServiceResult SetSetting(IDictionary<string, string> kvp);
    }

    public class SettingsService : ISettingsService
    {
        private readonly ILogger<SettingsService> _logger;
        private readonly string _settingsPath;
        private readonly string _defaultLastSavePath;
        private readonly string _defaultLastSaveReportPath;
        private readonly string _elementsFileName;
        private readonly string _elementsFullPath;

        public SettingsModel Settings { get; set; }

        public SettingsService(ILogger<SettingsService> logger)
        {
            _logger = logger;
            _settingsPath = Environment.ExpandEnvironmentVariables("%userprofile%/AppData/Local/Microdyn-Nadir-ROAM/settings.json");
            _defaultLastSavePath = Environment.ExpandEnvironmentVariables("%UserProfile%/Documents");
            _defaultLastSaveReportPath = Environment.ExpandEnvironmentVariables("%UserProfile%/Documents");
            _elementsFileName = "elements.json";
            _elementsFullPath = Environment.ExpandEnvironmentVariables($"%userprofile%/AppData/Local/Microdyn-Nadir-ROAM/{_elementsFileName}");
            Settings = new SettingsModel();
        }

        public ServiceResult SetSetting(IDictionary<string, string> settings)
        {
            KeyValuePair<string, string> settingHolder = new KeyValuePair<string, string>();
            try
            {
                foreach (var setting in settings)
                {
                    settingHolder = setting;
                    switch (setting.Key.ToLower())
                    {
                        case "lastprojectionby":
                            Settings.LastProjectionBy = setting.Value;
                            break;
                        case "lastsavepath":
                            Settings.LastSavePath = setting.Value;
                            break;
                        case "lastsavereportpath":
                            Settings.LastSaveReportPath = setting.Value;
                            break;
                        case "loglevel":
                            Settings.LogLevel = setting.Value;
                            break;
                        case "logretentiondays":
                            Settings.LogRetentionDays = Convert.ToInt32(setting.Value);
                            break;
                        case "iterationtimeoutseconds":
                            Settings.IterationTimeoutSeconds = Convert.ToInt32(setting.Value);
                            break;
                    }
                }
            }
            catch (Exception ex)
            {
                _logger.LogError(ex, $"SetSetting(): Exception setting '{settingHolder.Key}' with '{settingHolder.Value}'.");
                return new ServiceResult(Result.Error);
            }

            SaveSettings();

            return new ServiceResult(Result.Success);
        }

        public ServiceResult LoadSettings()
        {
            _logger.LogInformation($"LoadSettings(): Entered. Path='{_settingsPath}'");

            if (!File.Exists(_settingsPath))
            {
                _logger.LogInformation("LoadSettings(): Settings file doesn't exist");

                var saveResult = SaveSettings();
                if (saveResult.Result != Result.Success)
                    return saveResult;
            }

            try
            {
                using (var sr = new StreamReader(_settingsPath))
                {
                    var json = sr.ReadToEnd();
                    Settings = JsonConvert.DeserializeObject<SettingsModel>(json);
                }

                var loadElementsResult = LoadElements();
                if (loadElementsResult.Result != Result.Success)
                    throw new Exception($"Elements file is not valid.");

                Settings.Elements = (Dictionary<string, ElementModel>)loadElementsResult.Data;

                var validationResult = ValidateSettings();
                if (validationResult.Result != Result.Success)
                    throw new Exception($"Settings file is not valid: {validationResult.Data}");
            }
            catch (Exception ex)
            {
                _logger.LogCritical(ex, $"LoadSettings(): Exception reading settings file.");
                return new ServiceResult(Result.Critical);
            }
            _logger.LogInformation($"LoadSettings(): IterationsTimeoutSeconds = {Settings.IterationTimeoutSeconds}");
            _logger.LogInformation("LoadSettings(): Success");
            return new ServiceResult(Result.Success);
        }

        public ServiceResult SaveSettings()
        {
            _logger.LogInformation($"SaveSettings(): Entered. Path = '{_settingsPath}'");

            try
            {
                using (var sw = new StreamWriter(_settingsPath, false))
                {
                    var data = JObject.FromObject(Settings);
                    data.Remove("Elements");
                    var json = data.ToString(Formatting.Indented, new JsonConverter[] { new StringEnumConverter() });
                    //var json = JsonConvert.SerializeObject(Settings, Formatting.Indented, new JsonConverter[] { new StringEnumConverter() });

                    sw.Write(json);
                }
            }
            catch (Exception ex)
            {
                _logger.LogError(ex, $"SaveSettings(): Exception writing settings file.");
                return new ServiceResult(Result.Error);
            }

            _logger.LogInformation("SaveSettings(): Success");
            return new ServiceResult(Result.Success);
        }

        // Set default values for missing settings.  
        // Returns a critical error if a required setting is missing.
        private ServiceResult ValidateSettings()
        {
            var result = new ServiceResult(Result.Success);
            var data = "";
            var needsSaving = false;

            if (Settings == null)
                Settings = new SettingsModel();

            if (string.IsNullOrWhiteSpace(Settings.LastSavePath))
            {
                Settings.LastSavePath = _defaultLastSavePath;
                needsSaving = true;
            }

            if (string.IsNullOrWhiteSpace(Settings.LastSaveReportPath))
            {
                Settings.LastSaveReportPath = _defaultLastSaveReportPath;
                needsSaving = true;
            }

            try
            {
                NLog.LogLevel.FromString(Settings.LogLevel);
            }
            catch (Exception ex)
            {
                Settings.LogLevel = NLog.LogLevel.Info.ToString();
                needsSaving = true;
            }

            if (Settings.LogRetentionDays < 1)
            {
                Settings.LogRetentionDays = 5;
                needsSaving = true;
            }

            if (Settings.IterationTimeoutSeconds < 1)
            {
                Settings.IterationTimeoutSeconds = 10;
                needsSaving = true;
            }

            if (needsSaving)
                SaveSettings();

            // If any setting check above sets data with error string return a critical error
            if (!string.IsNullOrWhiteSpace(data))
            {
                result.Result = Result.Critical;
                result.Data = data;
            }

            return result;
        }

        public ServiceResult LoadElements()
        {
            _logger.LogInformation($"LoadElements(): Entered. Path='{_elementsFullPath}'");
            
            if (!File.Exists(_elementsFullPath))
            {
                _logger.LogInformation("LoadElements(): Settings file doesn't exist");

                // Check if file exists in same folder, usual case on build/publish
                _logger.LogInformation($"Current Dir: {AppDomain.CurrentDomain.BaseDirectory}");
                if (File.Exists($"{AppDomain.CurrentDomain.BaseDirectory}/{_elementsFileName}"))
                {
                    File.Move($"{AppDomain.CurrentDomain.BaseDirectory}/{_elementsFileName}", _elementsFullPath);
                }
                else // Otherwise, create default elements file
                {
                    var saveResult = SaveElements();
                    if (saveResult.Result != Result.Success)
                        return saveResult;
                }
            }

            var elements = new Dictionary<string, ElementModel>();
            try
            {
                using (var sr = new StreamReader(_elementsFullPath))
                {
                    var json = sr.ReadToEnd();
                    elements = JsonConvert.DeserializeObject<Dictionary<string, ElementModel>>(json);
                }
            }
            catch (Exception ex)
            {
                _logger.LogCritical(ex, $"LoadElements(): Exception reading elements file.");
                return new ServiceResult(Result.Critical);
            }

            _logger.LogInformation("LoadElements(): Success");
            return new ServiceResult(Result.Success, elements);
        }

        // Used if elements.json file is not found
        private ServiceResult SaveElements()
        {
            _logger.LogInformation($"SaveElements(): Entered. Path = '{_elementsFullPath}'");

            try
            {
                using (var sw = new StreamWriter(_elementsFullPath, false))
                {
                    var json = JsonConvert.SerializeObject(DefaultElements(), Formatting.Indented, new JsonConverter[] { new StringEnumConverter() });
                    sw.Write(json);
                }
            }
            catch (Exception ex)
            {
                _logger.LogError(ex, $"SaveElements(): Exception writing elements file.");
                return new ServiceResult(Result.Error);
            }

            _logger.LogInformation("SaveElements(): Success");
            return new ServiceResult(Result.Success);
        }

        // Default list of elements, only used if elements.json is missing from Visual Studio project.
        private Dictionary<string, ElementModel> DefaultElements()
        {
            var elements = new Dictionary<string, ElementModel>();

            var element = new ElementModel();
            element.Model = "4040-BW";
            element.Diameter = 4;
            element.Length = 40;
            element.Reject = 99.5;
            element.Flow = 2400;
            element.pcoef = 0.12;
            element.pExp = 1.35;
            element.area = 85;
            element.leak = 0.001;
            element.Atc1 = 9.902317;
            element.PSTD = 198;
            element.TCor = 2700;
            elements.Add(element.Model, element);

            element = new ElementModel();
            element.Model = "4040-LE";
            element.Diameter = 4;
            element.Length = 40;
            element.Reject = 99.3;
            element.Flow = 3350;
            element.pcoef = 0.12;
            element.pExp = 1.35;
            element.area = 85;
            element.leak = 0.001;
            element.Atc1 = 13.82198;
            element.PSTD = 198;
            element.TCor = 2700;
            elements.Add(element.Model, element);

            element = new ElementModel();
            element.Model = "4040-FR";
            element.Diameter = 4;
            element.Length = 40;
            element.Reject = 99.3;
            element.Flow = 1952;
            element.pcoef = 0.12;
            element.pExp = 1.35;
            element.area = 75;
            element.leak = 0.001;
            element.Atc1 = 9.127736;
            element.PSTD = 198;
            element.TCor = 2700;
            elements.Add(element.Model, element);

            element = new ElementModel();
            element.Model = "4040-XLE";
            element.Diameter = 4;
            element.Length = 40;
            element.Reject = 99.2;
            element.Flow = 5611;
            element.pcoef = 0.189;
            element.pExp = 1.49;
            element.area = 85;
            element.leak = 0.001;
            element.Atc1 = 23.15079;
            element.PSTD = 198;
            element.TCor = 2700;
            elements.Add(element.Model, element);

            element = new ElementModel();
            element.Model = "4040-NF9";
            element.Diameter = 4;
            element.Length = 40;
            element.Reject = 98.0;
            element.Flow = 3600;
            element.pcoef = 0.2;
            element.pExp = 1.45;
            element.area = 85;
            element.leak = 0.002;
            element.Atc1 = 30.01008;
            element.PSTD = 98;
            element.TCor = 2700;
            elements.Add(element.Model, element);

            element = new ElementModel();
            element.Model = "8040-BW-365";
            element.Diameter = 7.9;
            element.Length = 40;
            element.Reject = 99.5;
            element.Flow = 9800;
            element.pcoef = 0.0124;
            element.pExp = 1.56;
            element.area = 365;
            element.leak = 0.001;
            element.Atc1 = 9.416245;
            element.PSTD = 198;
            element.TCor = 2700;
            elements.Add(element.Model, element);

            element = new ElementModel();
            element.Model = "8040-BW-400";
            element.Diameter = 7.9;
            element.Length = 40;
            element.Reject = 99.5;
            element.Flow = 10700;
            element.pcoef = 0.0124;
            element.pExp = 1.56;
            element.area = 400;
            element.leak = 0.001;
            element.Atc1 = 9.381414;
            element.PSTD = 198;
            element.TCor = 2700;
            elements.Add(element.Model, element);

            element = new ElementModel();
            element.Model = "8040-BW-440";
            element.Diameter = 7.9;
            element.Length = 40;
            element.Reject = 99.5;
            element.Flow = 11700;
            element.pcoef = 0.037;
            element.pExp = 1.35;
            element.area = 440;
            element.leak = 0.001;
            element.Atc1 = 9.32562;
            element.PSTD = 198;
            element.TCor = 2700;
            elements.Add(element.Model, element);

            element = new ElementModel();
            element.Model = "8040-LE-400";
            element.Diameter = 7.9;
            element.Length = 40;
            element.Reject = 99.3;
            element.Flow = 16000;
            element.pcoef = 0.0124;
            element.pExp = 1.56;
            element.area = 400;
            element.leak = 0.001;
            element.Atc1 = 14.02828;
            element.PSTD = 198;
            element.TCor = 2700;
            elements.Add(element.Model, element);

            element = new ElementModel();
            element.Model = "8040-LE-440";
            element.Diameter = 7.9;
            element.Length = 40;
            element.Reject = 99.3;
            element.Flow = 17539;
            element.pcoef = 0.037;
            element.pExp = 1.35;
            element.area = 440;
            element.leak = 0.001;
            element.Atc1 = 13.97966;
            element.PSTD = 198;
            element.TCor = 2700;
            elements.Add(element.Model, element);

            element = new ElementModel();
            element.Model = "8040-FR-400/34";
            element.Diameter = 7.9;
            element.Length = 40;
            element.Reject = 99.4;
            element.Flow = 10250;
            element.pcoef = 0.0124;
            element.pExp = 1.56;
            element.area = 400;
            element.leak = 0.001;
            element.Atc1 = 8.986869;
            element.PSTD = 198;
            element.TCor = 2700;
            elements.Add(element.Model, element);            

            element = new ElementModel();
            element.Model = "8040-XLE-440";
            element.Diameter = 7.9;
            element.Length = 40;
            element.Reject = 99.2;
            element.Flow = 28484;
            element.pcoef = 0.037;
            element.pExp = 1.35;
            element.area = 440;
            element.leak = 0.001;
            element.Atc1 = 22.7035;
            element.PSTD = 198;
            element.TCor = 2700;
            elements.Add(element.Model, element);

            element = new ElementModel();
            element.Model = "8040-NF9-400";
            element.Diameter = 7.9;
            element.Length = 40;
            element.Reject = 98;
            element.Flow = 17100;
            element.pcoef = 0.0124;
            element.pExp = 1.56;
            element.area = 400;
            element.leak = 0.001;
            element.Atc1 = 30.29143;
            element.PSTD = 98;
            element.TCor = 2700;
            elements.Add(element.Model, element);
            return elements;
        }
    }
}
