﻿using System;
using System.Collections.Generic;
using System.Text;

using Microsoft.Extensions.Logging;

using ServiceApp.Enums;
using ServiceApp.Models;

namespace ServiceApp.Services
{
    public interface IRawWaterService
    {
        ServiceResult Calculate(ProjectionModel model);
        ServiceResult AutoBalance(ProjectionModel model);
        void CalculateSummation(ProjectionModel model);
        void CalculateOsmoIon(double tds, double[] vppm, Action<double> dIon_str, double temp, Action<double> OsmoticP);
        void CalculateSaturation(double temp, double pH, double tds, double[] vppm, double dIon_str, Action<double> BariumSulfateSat,
           Action<double> CaSulfateSat, Action<double> CaFluorideSat, Action<double> StronSulfateSat, Action<double> SilicaSat,
           Action<double> dIndexl, Action<double> dIndexs);
    }
    public class RawWaterService : IRawWaterService
    {
        private ILogger<RawWaterService> _logger { get; set; }
        private IComputationService _computationService { get; set; }

        private List<IonUnitConversionModel> _IonUnitConvFactors { get; set; }
        private List<MolecularWeightModel> _MolecularWeights;

        public RawWaterService(ILogger<RawWaterService> logger, IComputationService computationService)
        {
            _logger = logger;
            _computationService = computationService;
            _IonUnitConvFactors = SetIonUnitConversionFactors();
            _MolecularWeights = SetMolecularWeights();
        }

        public ServiceResult Calculate(ProjectionModel model)
        {
            // MH - The VB6 app saved data in Access as single precision float (only 7 significant digits).
            //      Adjusting the raw water values in a similar way so calulations match between the apps.
            //      Client may want the numbers to be more accurate in the end, simply comment this call out.
            _computationService.AdjustForTruncation(model);

            model.SetIonTypes();

            var origTemp = model.Temperature;
            model.Temperature = model.TemperatureUnit == "C" ? model.Temperature : (model.Temperature - 32) / 1.8;

            model.Ions = ConvertIonsToPPM(model.Ions);

            CalculateCO2CO3(model);
            CalculateSummation(model);
            model.WaterIsBalanced = CalculateBalance(model);
            if (!model.WaterIsBalanced)
            {
                model.Temperature = origTemp;
                return new ServiceResult(Result.Success, model);
            }

            CalculateOsmoIon(model.TDS, model.VPPM.ToArray(), val => model.ionStrength = val, model.Temperature, val => model.OsmoticPressure = val);
            CalculateSaturation(model.Temperature, model.pH, model.TDS, model.VPPM.ToArray(), model.ionStrength, val => model.KspBaSO4 = val, 
                val => model.KspCaSO4 = val, val => model.KspCaF2 = val, val => model.KspSrSO4 = val, val => model.SilicaSat = val,
                val => model.LSI = val, val => model.SDSI = val);

            model.Temperature = origTemp;


            return new ServiceResult(Result.Success, model);
        }

        public ServiceResult AutoBalance(ProjectionModel model)
        {
            // MH - The VB6 app saved data in Access as single precision float (only 7 significant digits).
            //      Adjusting the raw water values in a similar way so calulations match between the apps.
            //      Client may want the numbers to be more accurate in the end, simply comment this call out.
            _computationService.AdjustForTruncation(model);

            model.SetIonTypes();

            var origTemp = model.Temperature;
            model.Temperature = model.TemperatureUnit == "C" ? model.Temperature : (model.Temperature - 32) / 1.8;

            CalculateCO2CO3(model);
            CalculateSummation(model);

            double diffIons = Math.Abs(model.CatMeq - model.AnMeq);

            if (model.CatMeq > model.AnMeq)
                model.Ions[10].Value += 35.5 * diffIons;
            else
                model.Ions[2].Value += 23 * diffIons;

            CalculateSummation(model);
            model.WaterIsBalanced = CalculateBalance(model);
            if (!model.WaterIsBalanced)
                return new ServiceResult(Result.Success, model);

            CalculateOsmoIon(model.TDS, model.VPPM.ToArray(), val => model.ionStrength = val, model.Temperature, val => model.OsmoticPressure = val);
            CalculateSaturation(model.Temperature, model.pH, model.TDS, model.VPPM.ToArray(), model.ionStrength, val => model.KspBaSO4 = val,
                val => model.KspCaSO4 = val, val => model.KspCaF2 = val, val => model.KspSrSO4 = val, val => model.SilicaSat = val,
                val => model.LSI = val, val => model.SDSI = val);
            model.WaterIsBalanced = true;
            model.IsModified = true;

            model.Temperature = origTemp;
            _computationService.AdjustForTruncation(model);  // Since calculations were made adjust again
            return new ServiceResult(Result.Success, model);
        }

        public void CalculateSaturation(double temp, double pH, double tds, double[] vppm, double dIon_str, Action<double> BariumSulfateSat, 
            Action<double> CaSulfateSat, Action<double> CaFluorideSat, Action<double> StronSulfateSat, Action<double> SilicaSat,
            Action<double> dIndexl, Action<double> dIndexs)
        {
            if (temp == 0)
                temp = 25;

            var working = ((Math.Pow(10, -4)) * vppm[0] * vppm[9]) / (double)(_MolecularWeights.Find(mw => mw.Name.Equals("Calcium")).Weight *
                _MolecularWeights.Find(mw => mw.Name.Equals("Sulfate")).Weight) / (double)((Math.Pow((temp / (double)25), 0.152)) * (1.8 * Math.Pow(10, (-3))) *
                Math.Pow(dIon_str, 0.75));
            if(double.IsNaN(working))
                CaSulfateSat(0.0);
            else
                CaSulfateSat(working);

            working = ((Math.Pow(10, -4)) * vppm[5] * vppm[9]) / (double)(_MolecularWeights.Find(mw => mw.Name.Equals("Barium")).Weight *
                _MolecularWeights.Find(mw => mw.Name.Equals("Sulfate")).Weight) / (double)((Math.Pow((temp / (double)25), 0.625)) * (6.955 * Math.Pow(10, (-9))) *
                Math.Pow(dIon_str, 0.79));
            if (double.IsNaN(working))
                BariumSulfateSat(0.0);
            else
                BariumSulfateSat(working);

            working = ((Math.Pow(10, -4)) * vppm[6] * vppm[9]) / (double)(_MolecularWeights.Find(mw => mw.Name.Equals("Strontium")).Weight *
                _MolecularWeights.Find(mw => mw.Name.Equals("Sulfate")).Weight) / (double)((Math.Pow((temp / (double)25), 0.147)) * (1.325 * Math.Pow(10, (-5))) *
                Math.Pow(dIon_str, 0.76));
            if (double.IsNaN(working))
                StronSulfateSat(0.0);
            else
                StronSulfateSat(working);

            working = (100.0 * vppm[13] / (75.0 + 2.0 * temp)) / (double)((0.0479 * (Math.Pow(pH, 3))) - (0.8337 * (Math.Pow(pH, 2))) + (4.6244 * pH) - 6.9904);
            if (double.IsNaN(working))
                SilicaSat(0.0);
            else
                SilicaSat(working);

            working = 100.0 * ((vppm[0] / (double)(_MolecularWeights.Find(mw => mw.Name.Equals("Calcium")).Weight * 1000)) *
                Math.Pow((vppm[11] / (double)(_MolecularWeights.Find(mw => mw.Name.Equals("Fluoride")).Weight * 1000)), 2)) / (double)((double)3 * Math.Pow(10, (-10)) *
                (Math.Pow(dIon_str, 0.4128)));
            if (double.IsNaN(working))
                CaFluorideSat(0.0);
            else
                CaFluorideSat(working); 

            if ((vppm[0] == 0) || (vppm[8] == 0 && vppm[7] == 0) || pH == 0)
            {
                dIndexl(0);
                dIndexs(0);
            }
            else
            {
                var palk = 4.7 - Math.Log10((0.82 * vppm[8]) + (1.667 * vppm[7]));
                var pca = 5 - Math.Log10(2.5 * vppm[0]);
                var kindL = (Math.Log10(tds) / 13) + 1.95 + (0.012 * (48 - (1.8 * temp)));
                var Kinds50 = 2.804 - (1.21 * (Math.Pow(10, (-3))) * Math.Exp(5 * (1.4 - dIon_str)));
                var Kinds30 = 3.308 - (7.16 * (Math.Pow(10, (-3))) * Math.Exp(3.75 * (1.4 - dIon_str)));
                var Kinds0 = 3.832 - (2.96 * (Math.Pow(10, (-2))) * Math.Exp(2.5 * (1.4 - dIon_str)));
                double kinds = 0;

                if (temp >= 50)
                    kinds = Kinds50;
                else if (temp == 30)
                    kinds = Kinds30;
                else if (temp < -0)
                    kinds = Kinds0;
                else if (temp < 30)
                    kinds = Kinds0 - (temp * (Kinds0 - Kinds30) / 30);
                else if (temp > 30)
                    kinds = Kinds50 + (((Kinds30 - Kinds50) / 20) * (50 - temp));

                dIndexl(pH - palk - pca - kindL);
                dIndexs(pH - palk - pca - kinds);
            }            
        }

        public void CalculateOsmoIon(double tds, double[] vppm, Action<double> dIon_str, double temp, Action<double> OsmoticP)
        {
            double mol1Sum = 0;
            double sum = 0;

            for (int i = 0; i < 14; i++)
            {
                var step1 = ((double)1 - tds / (double)1000000);
                var step2 = _MolecularWeights[i].Weight * step1;
                var step3 = vppm[i] / step2;
                var step4 = Math.Pow(10, -3) * step3;
                var mol1 = (Math.Pow(10, -3) * (vppm[i] / (_MolecularWeights[i].Weight * ((double)1 - tds / (double)1000000))));  // C# calc wrong value without all the parens, even though it should be equiv. without them.
                mol1Sum += mol1;
                sum += mol1 * Math.Pow(_MolecularWeights[i].Valency, 2);
            }

            dIon_str(0.5 * sum);          

            OsmoticP(1.12 * (temp + 273) * mol1Sum);
        }

        private bool CalculateBalance(ProjectionModel model)
        {
            var isBalanced = true;
            //model.IsModified = true;

            double diffIons = Math.Abs(model.CatMeq - model.AnMeq);

            if (model.CatMeq != model.AnMeq && (diffIons / model.CatMeq) > 0.1)
            {
                isBalanced = false;
            }

            return isBalanced;
        }

        public void CalculateSummation(ProjectionModel model)
        {
            model.CationPpm = 0;
            model.CationCacO3 = 0;
            model.CatMeq = 0;
            model.AnionPpm = 0;
            model.AnionCacO3 = 0;
            model.AnMeq = 0;

            for(int i = 0; i <= 13; i++)
            {
                if (model.Ions[i].UnitType == "ppm")
                {
                    model.VPPM[i] = model.Ions[i].Value;
                    model.VCacO3[i] = model.Ions[i].Value / _IonUnitConvFactors[i].CaCO3;
                    model.VMilli[i] = model.Ions[i].Value / _IonUnitConvFactors[i].meqL;
                }
                else if (model.Ions[i].UnitType == "CaCO3")
                {
                    model.VPPM[i] = model.Ions[i].Value * _IonUnitConvFactors[i].CaCO3;
                    model.VCacO3[i] = model.Ions[i].Value;
                    model.VMilli[i] = model.Ions[i].Value / _IonUnitConvFactors[i].meqL;
                }
                else  // meq/L
                {
                    model.VPPM[i] = model.Ions[i].Value * _IonUnitConvFactors[i].meqL;
                    model.VCacO3[i] = model.Ions[i].Value * _IonUnitConvFactors[i].CaCO3;
                    model.VMilli[i] = model.Ions[i].Value;
                }

                model.Ions[i].Weight = model.VMilli[i];

                if (model.Ions[i].IonType == IonType.Cation)
                {
                    model.CationPpm += model.VPPM[i];
                    model.CationCacO3 += model.VCacO3[i];
                    model.CatMeq += model.VMilli[i];
                }
                else if(i < 13) // Don't add silica here
                {
                    model.AnionPpm += model.VPPM[i];
                    model.AnionCacO3 += model.VCacO3[i];
                    model.AnMeq += model.VMilli[i];
                }               
            }

            model.TDS = model.CationPpm + model.AnionPpm + model.VPPM[13];
            model.CationsTotal = model.CatMeq;
            model.AnionsTotal = model.AnMeq;
        }

        private void CalculateCO2CO3(ProjectionModel model)
        {
            double CO3 = model.Ions[7].Value;
            double pH = model.pH;
            double HCO3 = model.Ions[8].Value;
            double CO2 = 0.0;
            double CO2b = 0.0;
            double CO3b = 0.0;

            // MH - Verify with client: There is a line in original VB that sets this again
            CO3 = 0;

            HCO3 = HCO3 * 0.82;
            double k1 = 4.4 * Math.Pow(10, -7);
            double k2 = 4.69 * Math.Pow(10, -11);
            double h = Math.Pow(10, pH * -1);
            double hexp = ((h / k1) + (k2 / h) + 1);
            double ta = 0;

            if (pH < 8.3 || CO3 < 0.1)
                ta = HCO3 / (double)50000 * hexp;
            else
                ta = (HCO3 / (double)50000) + (CO3 / (double)100000);

            if (ta > 0)
            {
                CO2 = Math.Pow(10, ((Math.Log10(ta) - Math.Log10(hexp) + Math.Log10(h / k1)))) * (double)44000;
                CO3b = Math.Pow(10, ((Math.Log10(ta) - Math.Log10(hexp) + Math.Log10(k2 / h)))) * (double)100000;
            }
            else
            {
                CO2 = 0;
                CO3b = 0;
            }

            if (Math.Abs(CO2) < 0.1)
            {
                if (Math.Abs(CO2) < 0.05)
                    CO2b = 0;
                else
                    CO2b = 0.1;
            }
            else
            {
                CO2b = CO2;
            }

            if (CO3b != 0)
            {
                if (Math.Abs(CO3 - (CO3b / 1.667)) > (double)1)
                {
                    if ((double)2 * Math.Abs(CO3 - (CO3b / 1.667)) / (CO3 + (CO3b / 1.667)) > 0.05)
                        CO3 = CO3b / 1.667;
                }
            }

            // model.CarbonDioxide = CO2b; // MH testing if this should be CO2b instead of CO2
            model.CarbonDioxide = CO2; // MH testing if this should be CO2b instead of CO2
            model.Ions[7].Value = CO3;
            model.Ions[8].Value = HCO3 / 0.82;
        }

        private List<IonModel> ConvertIonsToPPM(List<IonModel> ions)
        {
            var ppmIons = new List<IonModel>();
            foreach (IonModel ion in ions)
            {                
                var ppmIon = new IonModel(ion.Name, ion.IonType, 0.0, "ppm");
                switch (ion.UnitType)
                {
                    case "CaCO3":
                        ppmIon.Value = ion.Value * _IonUnitConvFactors.Find(iucf => iucf.Name.Equals(ion.Name)).CaCO3;
                        break;
                    case "meq/L":
                        ppmIon.Value = ion.Value * _IonUnitConvFactors.Find(iucf => iucf.Name.Equals(ion.Name)).meqL;
                        break;
                    default:
                        ppmIon.Value = ion.Value; // already PPM
                        break;
                }

                ppmIons.Add(ppmIon);
            }

            return ppmIons;
        }

        private List<MolecularWeightModel> SetMolecularWeights()
        {
            var weights = new List<MolecularWeightModel>();

            weights.Add(new MolecularWeightModel() { Name = "Calcium", Weight = 40.1, Valency = 2 });
            weights.Add(new MolecularWeightModel() { Name = "Magnesium", Weight = 24.3, Valency = 2 });
            weights.Add(new MolecularWeightModel() { Name = "Sodium", Weight = 23.0, Valency = 1 });
            weights.Add(new MolecularWeightModel() { Name = "Potassium", Weight = 39.0, Valency = 1 });
            weights.Add(new MolecularWeightModel() { Name = "Ammonium", Weight = 18.0, Valency = 1 });
            weights.Add(new MolecularWeightModel() { Name = "Barium", Weight = 137.3, Valency = 2 });
            weights.Add(new MolecularWeightModel() { Name = "Strontium", Weight = 87.6, Valency = 2 });
            weights.Add(new MolecularWeightModel() { Name = "Carbonate", Weight = 60.0, Valency = -2 });
            weights.Add(new MolecularWeightModel() { Name = "Bicarbonate", Weight = 61.0, Valency = -1 });
            weights.Add(new MolecularWeightModel() { Name = "Sulfate", Weight = 96.0, Valency = -2 });
            weights.Add(new MolecularWeightModel() { Name = "Chloride", Weight = 35.45, Valency = -1 });
            weights.Add(new MolecularWeightModel() { Name = "Fluoride", Weight = 19.0, Valency = -1 });
            weights.Add(new MolecularWeightModel() { Name = "Nitrate", Weight = 62.0, Valency = -1 });
            weights.Add(new MolecularWeightModel() { Name = "Silica", Weight = 60.1, Valency = -1 });

            return weights;
        }

        private List<IonUnitConversionModel> SetIonUnitConversionFactors()
        {
            var factors = new List<IonUnitConversionModel>();
            factors.Add(new IonUnitConversionModel() { Name = "Calcium", meqL = 20.04, CaCO3 = 0.4 });
            factors.Add(new IonUnitConversionModel() { Name = "Magnesium", meqL = 12.16, CaCO3 = 0.24 });
            factors.Add(new IonUnitConversionModel() { Name = "Sodium", meqL = 22.99, CaCO3 = 0.46 });
            factors.Add(new IonUnitConversionModel() { Name = "Potassium", meqL = 39.1, CaCO3 = 0.78 });
            factors.Add(new IonUnitConversionModel() { Name = "Ammonium", meqL = 18.04, CaCO3 = 0.36 });
            factors.Add(new IonUnitConversionModel() { Name = "Barium", meqL = 68.67, CaCO3 = 1.37 });
            factors.Add(new IonUnitConversionModel() { Name = "Strontium", meqL = 43.81, CaCO3 = 0.88 });
            factors.Add(new IonUnitConversionModel() { Name = "Carbonate", meqL = 30, CaCO3 = 0.6});
            factors.Add(new IonUnitConversionModel() { Name = "Bicarbonate", meqL = 61.01, CaCO3 = 1.22});
            factors.Add(new IonUnitConversionModel() { Name = "Sulfate", meqL = 48.03, CaCO3 = 0.96});
            factors.Add(new IonUnitConversionModel() { Name = "Chloride", meqL = 35.45, CaCO3 = 0.71});
            factors.Add(new IonUnitConversionModel() { Name = "Fluoride", meqL = 19, CaCO3 = 0.38});
            factors.Add(new IonUnitConversionModel() { Name = "Nitrate", meqL = 62, CaCO3 = 1.24});
            factors.Add(new IonUnitConversionModel() { Name = "Silica", meqL = 60.08, CaCO3 = 1.2});

            return factors;
        }
    }
}
