﻿using Microsoft.Extensions.Logging;
using ServiceApp.Enums;
using ServiceApp.Models;
using System;
using System.Collections.Generic;
using System.Diagnostics;

namespace ServiceApp.Services
{
    public interface IDesignCalculationService
    {
        ServiceResult Init(ProjectionModel model);
        ServiceResult Calc(ProjectionModel model);
    }
    
    public class DesignCalculationService : IDesignCalculationService
    {
        //private readonly int _WaitTime; // This is the time, in seconds, the program can run before it is decided the program does not converge. Now set in settings.
        private const int _iArray = 7;
        private const double _PSItoBAR = 0.068947;
        private const double _GPMtoM3 = 0.2271;
        private const double _GFDtoLMH = 1.698;
        private const int _SB_Type = 1;
        private const int _A153_TS80_Type = 2;
        private const int _A152_ACM1_Type = 3;
        private const int _Other_Type = 4;
        private const int _TS_40 = 5;
        private const int _UF_MF = 6;
        private const double _bexp40 = 0.9;
        private const double _bexp60 = 0.56;

        private Dictionary<string, ElementModel> _elements;

        private IRawWaterService _rawWaterService { get; set; }
        //private IComputationService _compService { get; set; }
        private IProjectionService _projectionService { get; set; }
        private ILogger<DesignCalculationService> _logger { get; set; }
        private ILogService _logService { get; set; }
        private ISettingsService _settingsService { get; set; }

        public DesignCalculationService(IRawWaterService rawWaterService, IProjectionService projectionService, ILogger<DesignCalculationService> logger,
            ILogService logService, ISettingsService settingsService)
        {
            _rawWaterService = rawWaterService;
            //_compService = compService;
            _projectionService = projectionService;
            _logger = logger;
            _logService = logService;
            _settingsService = settingsService;
            
            _logger.LogInformation($"IterationsTimeoutSeconds: {_settingsService.Settings.IterationTimeoutSeconds}");
        }

        public ServiceResult Init(ProjectionModel model)
        {
            model.Pass[0].FeedTemp = model.Temperature;
            model.Pass[0].FeedTempUnit = model.TemperatureUnit;
            model.Pass[0].RawWaterpH = model.pH;
            model.Pass[1].FeedTemp = model.Temperature;
            model.Pass[1].FeedTempUnit = model.TemperatureUnit;
            model.Pass[1].RawWaterpH = model.pH;

            return new ServiceResult(Result.Success, model);
        }

        public ServiceResult Calc(ProjectionModel model)
        {
            _logService.LogValues(model, "Entering Calc");

            ServiceResult serviceResult;
            
            _elements = _settingsService.Settings.Elements;

            if (model.EnablePass2)
            {
                serviceResult = Calc2Pass(model);
                if (serviceResult.Result != Result.Success)
                    return serviceResult;
            }
            else
            {
                if (model.Pass[0].RecircFlow == 0)
                {
                    serviceResult = Initialize(model, 0, 0);
                    if (serviceResult.Result != Result.Success)
                        return serviceResult;

                    serviceResult = Calc1Pass(model, 0);
                    if (serviceResult.Result != Result.Success)
                        return serviceResult;
                }
                else
                {
                    serviceResult = CalcWithRecirc(model);
                    if (serviceResult.Result != Result.Success)
                        return serviceResult;
                }

                model.Pass[0].CalcData.RecircFlowOriginal = 0;
            }
            model.bCalc = true;
            model.CalcDate = DateTime.Now;

            // MH - Store values for report
            var numPasses = model.EnablePass2 ? 2 : 1;
            for (int k = 0; k < numPasses; k++)
            {
                for (int j = 0; j < model.Pass[k].StageData.Count; j++)
                {
                    for (int i = 1; i <= model.Pass[k].StageData[j].ElementsVessel; i++)
                    {
                        var element = new ProjectionModel.ElementData();
                        element.ElementNumber = i;
                        element.PermeateFlow = model.Pass[k].CalcData.flowElem[j, i];
                        element.TDS = model.Pass[k].CalcData.tdsElem[j, i];
                        element.Beta = model.Pass[k].CalcData.betaElem[j, i];
                        element.Flux = model.Pass[k].CalcData.fluxElem[j, i];
                        element.NDPressure = model.Pass[k].CalcData.pnetiElem[j, i];
                        model.Pass[k].StageData[j].ElementData.Add(element);
                    }
                }
            }

            // MH - Water Saturation for report
            double feedTemp = model.Pass[0].FeedTempUnit == "C" ? model.Pass[0].FeedTemp : (model.Pass[0].FeedTemp - 32.0) / 1.8;            
            double tdsRaw = 0.0;
            double tdsFeed = 0.0;
            double tdsPass1Conc = 0.0;
            double tdsPass2Conc = 0.0;
            for (int i = 0; i < 14; i++)
            {
                tdsRaw += model.Pass[0].CalcData.ion_ppm[i];
                tdsFeed += model.Pass[0].CalcData.t_ion_ppm[i];
                tdsPass1Conc += model.Pass[0].CalcData.ion_ppmc[i];
                tdsPass2Conc += model.Pass[1].CalcData.ion_ppmc[i];
            }

            //_rawWaterService.CalculateSummation(model);
            _rawWaterService.CalculateOsmoIon(tdsRaw, model.Pass[0].CalcData.ion_ppm, val => model.ionStrength = val, feedTemp, val => model.RawWater.OsmoticPressure = val);
            _rawWaterService.CalculateSaturation(feedTemp, model.pH, tdsRaw, model.Pass[0].CalcData.ion_ppm, model.ionStrength,
                val => model.RawWater.KspBaSO4 = val, val => model.RawWater.KspCaSO4 = val, val => model.RawWater.KspCaF2 = val, val => model.RawWater.KspSrSO4 = val,
                val => model.RawWater.SilicaSat = val, val => model.RawWater.LSI = val, val => model.RawWater.SDSI = val);

            _rawWaterService.CalculateOsmoIon(tdsFeed, model.Pass[0].CalcData.t_ion_ppm, val => model.ionStrength = val, feedTemp, val => model.FeedWater.OsmoticPressure = val);
            _rawWaterService.CalculateSaturation(feedTemp, model.pH, tdsFeed, model.Pass[0].CalcData.t_ion_ppm, model.ionStrength,
                val => model.FeedWater.KspBaSO4 = val, val => model.FeedWater.KspCaSO4 = val, val => model.FeedWater.KspCaF2 = val, val => model.FeedWater.KspSrSO4 = val,
                val => model.FeedWater.SilicaSat = val, val => model.FeedWater.LSI = val, val => model.FeedWater.SDSI = val);

            _rawWaterService.CalculateOsmoIon(tdsPass1Conc, model.Pass[0].CalcData.ion_ppmc, val => model.ionStrength = val, feedTemp, val => model.Pass1Conc.OsmoticPressure = val);
            _rawWaterService.CalculateSaturation(feedTemp, model.Pass[0].CalcData.cph, tdsPass1Conc, model.Pass[0].CalcData.ion_ppmc, model.ionStrength,
                val => model.Pass1Conc.KspBaSO4 = val, val => model.Pass1Conc.KspCaSO4 = val, val => model.Pass1Conc.KspCaF2 = val, val => model.Pass1Conc.KspSrSO4 = val,
                val => model.Pass1Conc.SilicaSat = val, val => model.Pass1Conc.LSI = val, val => model.Pass1Conc.SDSI = val);

            _rawWaterService.CalculateOsmoIon(tdsPass2Conc, model.Pass[1].CalcData.ion_ppmc, val => model.ionStrength = val, feedTemp, val => model.Pass2Conc.OsmoticPressure = val);
            _rawWaterService.CalculateSaturation(feedTemp, model.Pass[1].CalcData.cph, tdsPass2Conc, model.Pass[1].CalcData.ion_ppmc, model.ionStrength,
                val => model.Pass2Conc.KspBaSO4 = val, val => model.Pass2Conc.KspCaSO4 = val, val => model.Pass2Conc.KspCaF2 = val, val => model.Pass2Conc.KspSrSO4 = val,
                val => model.Pass2Conc.SilicaSat = val, val => model.Pass2Conc.LSI = val, val => model.Pass2Conc.SDSI = val);

            // MH - Call dosage to get warnings for report
            var S = "";
            var t = "";
            var u = "";
            var v = "";
            var sTripol = "";
            var Base_Dose = 0.0;
            var Mix = 0.0;
            var Water = 0.0;
            var Daily_Usage = 0.0;
            var Month_Usage = 0.0;

            Dosage(model, 0, ref S, ref t, ref u, ref v, model.LSI, model.RawWater.KspCaSO4, model.RawWater.KspCaF2, model.RawWater.KspSrSO4, model.RawWater.KspBaSO4,
                model.RawWater.SilicaSat, ref sTripol, ref Base_Dose, ref Mix, ref Water, ref Daily_Usage, ref Month_Usage);
            model.Pass[0].Warnings = u;

            u = "";
            Dosage(model, 1, ref S, ref t, ref u, ref v, model.LSI, model.RawWater.KspCaSO4, model.RawWater.KspCaF2, model.RawWater.KspSrSO4, model.RawWater.KspBaSO4,
                model.RawWater.SilicaSat, ref sTripol, ref Base_Dose, ref Mix, ref Water, ref Daily_Usage, ref Month_Usage);
            model.Pass[1].Warnings = u;


            if (model.EnablePass2)
            {
                model.OverallSystemRecovery = 100.0 * model.Pass[1].ProductFlow / (model.Pass[0].StageData[0].FeedFlow - model.Pass[0].RecircFlow);
                //model.Pass[0].Recovery = model.Pass[0].ProductFlow * 100.0 / (model.Pass[0].StageData[0].FeedFlow - model.Pass[0].RecircFlow);
            }

            return new ServiceResult(Result.Success, model);
        }

        private ServiceResult Initialize(ProjectionModel model, int passNumber, int flag)
        {
            ResetData(model, passNumber);
            InitData(model, passNumber);

            // MH - The VB6 app saved data in Access as single precision float (only 7 significant digits).
            //      Adjusting the raw water values in a similar way so calulations match between the apps.
            //      Client may want the numbers to be more accurate in the end, simply comment this call out.            

            //            _compService.AdjustForTruncation(model);

            InitIons(model, flag);
            //          _compService.AdjustForTruncation(model);

            TempConv(model, passNumber);
            //        _compService.AdjustForTruncation(model);

            FlowConv(model, passNumber);
            //      _compService.AdjustForTruncation(model);

            InitFlow(model, passNumber);
            //    _compService.AdjustForTruncation(model);

            MembranInfo(model, passNumber);
            //  _compService.AdjustForTruncation(model);

            var serviceResult = InitPressure(model, passNumber);
            //_compService.AdjustForTruncation(model);           
            return serviceResult;
        }

        private void ResetData(ProjectionModel model, int passNumber)
        {
            model.Pass[passNumber].CalcData.fc022 = 0;
            model.Pass[passNumber].CalcData.Feed_Ret = 0;
            model.Pass[passNumber].CalcData.Con_Ret = 0;
            model.Pass[passNumber].CalcData.cph = 0;
            if (passNumber == 0)
                model.Pass[passNumber].CalcData.pph = 0;

            model.Pass[passNumber].CalcData.dos = 0;

            model.Pass[passNumber].CalcData.RetStatus = 0;

            model.Pass[passNumber].CalcData.cph2 = 0;
            model.Pass[passNumber].CalcData.pph2 = 0;
            model.Pass[passNumber].CalcData.age = 0;
            model.Pass[passNumber].CalcData.bfact = 0;
            model.Pass[passNumber].CalcData.ftemp = 0;
            model.Pass[passNumber].CalcData.TCor = 0;
            model.Pass[passNumber].CalcData.PProd = 0;
            model.Pass[passNumber].CalcData.sp_fact = 0;
            model.Pass[passNumber].CalcData.fee = 0;
            model.Pass[passNumber].CalcData.con = 0;
            model.Pass[passNumber].CalcData.fco3 = 0;
            model.Pass[passNumber].CalcData.fco2 = 0;
            model.Pass[passNumber].CalcData.fhco3 = 0;
            model.Pass[passNumber].CalcData.acids = 0;
            model.Pass[passNumber].CalcData.acidc = 0;
            model.Pass[passNumber].CalcData.caustic = 0;
            model.Pass[passNumber].CalcData.fcl = 0;
            model.Pass[passNumber].CalcData.fso4 = 0;
            model.Pass[passNumber].CalcData.fna = 0;
            model.Pass[passNumber].CalcData.prod_toti = 0;
            model.Pass[passNumber].CalcData.prod_totp = 0;
            model.Pass[passNumber].CalcData.deltapav = 0;
            model.Pass[passNumber].CalcData.osm_c = 0;
            model.Pass[passNumber].CalcData.feedav = 0;
            model.Pass[passNumber].CalcData.pnetn_av = 0;
            model.Pass[passNumber].CalcData.flow_ft = 0;
            model.Pass[passNumber].CalcData.rec_l = 0;
            model.Pass[passNumber].CalcData.rec_h = 0;
            model.Pass[passNumber].CalcData.rec_av = 0;
            model.Pass[passNumber].CalcData.pressf = 0;
            model.Pass[passNumber].CalcData.Rec = 0;
            model.Pass[passNumber].CalcData.p_salp = 0;
            model.Pass[passNumber].CalcData.p_prod = 0;
            model.Pass[passNumber].CalcData.flow_f = 0;
            model.Pass[passNumber].CalcData.flow_pt = 0;
            model.Pass[passNumber].CalcData.press_l = 0;
            model.Pass[passNumber].CalcData.press_h = 0;
            model.Pass[passNumber].CalcData.flow_b = 0;
            model.Pass[passNumber].CalcData.flow_bav = 0;
            model.Pass[passNumber].CalcData.deltap = 0;
            model.Pass[passNumber].CalcData.pneti = 0;
            model.Pass[passNumber].CalcData.prod = 0;
            model.Pass[passNumber].CalcData.feed = 0;
            model.Pass[passNumber].CalcData.sumfp = 0;
            model.Pass[passNumber].CalcData.fdecl = 0;
            model.Pass[passNumber].CalcData.avsp = 0;
            model.Pass[passNumber].CalcData.osma = 0;
            model.Pass[passNumber].CalcData.osmb = 0;
            model.Pass[passNumber].CalcData.tdsf = 0;
            model.Pass[passNumber].CalcData.avg_ppress = 0;
            model.Pass[passNumber].CalcData.ppress_sum = 0;
            model.Pass[passNumber].CalcData.ftemp2 = 0;
            model.Pass[passNumber].CalcData.pprod2 = 0;
            model.Pass[passNumber].CalcData.Blend = 0;
            model.Pass[passNumber].CalcData.recint = 0;
            model.Pass[passNumber].CalcData.start = 0;
            model.Pass[passNumber].CalcData.elapsed = 0;
            model.Pass[passNumber].CalcData.ph = 0;
            model.Pass[passNumber].CalcData.fph = 0;
            model.Pass[passNumber].CalcData.feedflow = 0;

            model.Pass[passNumber].CalcData.nbanks = 0;
            model.Pass[passNumber].CalcData.el_tot = 0;

            model.Pass[passNumber].CalcData.str = "";
            model.Pass[passNumber].CalcData.ptr = "";
            model.Pass[passNumber].CalcData.ptr2 = "";
            model.Pass[passNumber].CalcData.tdsc = 0;

            model.Pass[passNumber].CalcData.TotalArea = 0;
        }

        private void InitData(ProjectionModel model, int passNumber)
        {
            double unitAdj = model.Pass[0].UnitType.ToLower() == "metric" ? _PSItoBAR : 1.0;
            model.Pass[passNumber].CalcData.nbanks = model.Pass[passNumber].StageData.Count;

            for (int i = 0; i < model.Pass[passNumber].CalcData.nbanks; i++)
            {
                model.Pass[passNumber].CalcData.Tube[i] = model.Pass[passNumber].StageData[i].NumberPressureVessels;
                model.Pass[passNumber].CalcData.El_tu[i] = model.Pass[passNumber].StageData[i].ElementsVessel;
                model.Pass[passNumber].CalcData.Ppress[i] = model.Pass[passNumber].StageData[i].PermeatePressure / unitAdj;
                model.Pass[passNumber].CalcData.BoostPress[i] = (passNumber == 0 && i == 0) ? 0.0 : model.Pass[passNumber].StageData[i].BoostPressure / unitAdj;
            }
        }

        private void InitIons(ProjectionModel model, int flag)
        {
            var passNumber = 0;

            if (flag == 0 || flag == 3)
            {
                model.Pass[0].CalcData.t_ion_ppm[0] = model.Ions[0].Value;
                model.Pass[0].CalcData.ion_ppmf[0] = model.Ions[0].Value;
                model.Pass[0].CalcData.ion_ppm[0] = model.Ions[0].Value;
                model.Pass[0].CalcData.t_ion_ppm[1] = model.Ions[1].Value;
                model.Pass[0].CalcData.ion_ppmf[1] = model.Ions[1].Value;
                model.Pass[0].CalcData.ion_ppm[1] = model.Ions[1].Value;
                model.Pass[0].CalcData.t_ion_ppm[2] = model.Ions[2].Value;
                model.Pass[0].CalcData.ion_ppmf[2] = model.Ions[2].Value;
                model.Pass[0].CalcData.ion_ppm[2] = model.Ions[2].Value;
                model.Pass[0].CalcData.t_ion_ppm[3] = model.Ions[3].Value;
                model.Pass[0].CalcData.ion_ppmf[3] = model.Ions[3].Value;
                model.Pass[0].CalcData.ion_ppm[3] = model.Ions[3].Value;
                model.Pass[0].CalcData.t_ion_ppm[4] = model.Ions[4].Value;
                model.Pass[0].CalcData.ion_ppmf[4] = model.Ions[4].Value;
                model.Pass[0].CalcData.ion_ppm[4] = model.Ions[4].Value;
                model.Pass[0].CalcData.t_ion_ppm[5] = model.Ions[5].Value;
                model.Pass[0].CalcData.ion_ppmf[5] = model.Ions[5].Value;
                model.Pass[0].CalcData.ion_ppm[5] = model.Ions[5].Value;
                model.Pass[0].CalcData.t_ion_ppm[6] = model.Ions[6].Value;
                model.Pass[0].CalcData.ion_ppmf[6] = model.Ions[6].Value;
                model.Pass[0].CalcData.ion_ppm[6] = model.Ions[6].Value;
                model.Pass[0].CalcData.t_ion_ppm[7] = model.Ions[7].Value;
                model.Pass[0].CalcData.ion_ppmf[7] = model.Ions[7].Value;
                model.Pass[0].CalcData.ion_ppm[7] = model.Ions[7].Value;
                model.Pass[0].CalcData.t_ion_ppm[8] = model.Ions[8].Value;
                model.Pass[0].CalcData.ion_ppmf[8] = model.Ions[8].Value;
                model.Pass[0].CalcData.ion_ppm[8] = model.Ions[8].Value;
                model.Pass[0].CalcData.t_ion_ppm[9] = model.Ions[9].Value;
                model.Pass[0].CalcData.ion_ppmf[9] = model.Ions[9].Value;
                model.Pass[0].CalcData.ion_ppm[9] = model.Ions[9].Value;
                model.Pass[0].CalcData.t_ion_ppm[10] = model.Ions[10].Value;
                model.Pass[0].CalcData.ion_ppmf[10] = model.Ions[10].Value;
                model.Pass[0].CalcData.ion_ppm[10] = model.Ions[10].Value;
                model.Pass[0].CalcData.t_ion_ppm[11] = model.Ions[11].Value;
                model.Pass[0].CalcData.ion_ppmf[11] = model.Ions[11].Value;
                model.Pass[0].CalcData.ion_ppm[11] = model.Ions[11].Value;
                model.Pass[0].CalcData.t_ion_ppm[12] = model.Ions[12].Value;
                model.Pass[0].CalcData.ion_ppmf[12] = model.Ions[12].Value;
                model.Pass[0].CalcData.ion_ppm[12] = model.Ions[12].Value;
                model.Pass[0].CalcData.t_ion_ppm[13] = model.Ions[13].Value;
                model.Pass[0].CalcData.ion_ppmf[13] = model.Ions[13].Value;
                model.Pass[0].CalcData.ion_ppm[13] = model.Ions[13].Value;
            }

            if (flag == 2)
            {
                passNumber = 1;
                for (int k = 0; k <= 13; k++)
                {
                    model.Pass[1].CalcData.t_ion_ppm[k] = model.Pass[0].CalcData.ion_ppmp[k]; // MH - Check that right side is pass 1
                    model.Pass[1].CalcData.ion_ppmf[k] = model.Pass[0].CalcData.ion_ppmp[k];
                    model.Pass[1].CalcData.ion_ppm[k] = model.Pass[0].CalcData.ion_ppmp[k];
                }
            }

            if (flag == 3)
            {
                var BrinePass2 = model.Pass[0].ProductFlow - model.Pass[1].ProductFlow;
                var Ratio1 = (model.Pass[0].ProductFlow * 100 / model.Pass[0].Recovery - BrinePass2) / (model.Pass[0].ProductFlow * 100 / model.Pass[0].Recovery);
                var Ratio2 = BrinePass2 / (model.Pass[0].ProductFlow * 100 / model.Pass[0].Recovery);
                for (int k = 0; k <= 13; k++)
                {
                    model.Pass[0].CalcData.t_ion_ppm[k] = model.Pass[0].CalcData.ion_ppmf[k] * Ratio1 + model.Pass[1].CalcData.ion_ppmc[k] * Ratio2;
                    model.Pass[0].CalcData.ion_ppmf[k] = model.Pass[0].CalcData.ion_ppmf[k] * Ratio1 + model.Pass[1].CalcData.ion_ppmc[k] * Ratio2;
                }
            }

            double adjustedpH = 0.0;
            if (flag == 2)
            {
                adjustedpH = model.Pass[1].AdjustedpH == null ? 0.0 : (double)model.Pass[1].AdjustedpH;  
                model.Pass[1].RawWaterpH = model.Pass[0].CalcData.pph;
                model.Pass[1].CalcData.ph = model.Pass[0].CalcData.pph;
                model.Pass[1].CalcData.fph = adjustedpH > 0.0 ? adjustedpH : model.Pass[0].CalcData.pph;

                if (model.Pass[1].CalcData.fph < model.Pass[1].CalcData.ph && model.Pass[1].ChemType != "HCl")
                    model.Pass[1].ChemType = "H2SO4";

                if (model.Pass[1].CalcData.fph > model.Pass[1].CalcData.ph)
                    model.Pass[1].ChemType = "NaOH";
            }
            else
            {
                adjustedpH = model.Pass[0].AdjustedpH == null ? 0.0 : (double)model.Pass[0].AdjustedpH;
                model.Pass[0].CalcData.ph = model.pH;
                model.Pass[0].CalcData.fph = adjustedpH > 0.0 ? adjustedpH : model.Pass[0].RawWaterpH;

                if (model.Pass[0].CalcData.fph < model.Pass[0].CalcData.ph && model.Pass[0].ChemType != "HCl")
                    model.Pass[0].ChemType = "H2SO4";

                if (model.Pass[0].CalcData.fph > model.Pass[0].CalcData.ph)
                    model.Pass[0].ChemType = "NaOH";
            }

            var chemType = model.Pass[1].ChemType;
            if (flag == 0 || flag == 3)
                chemType = model.Pass[0].ChemType;



            if (chemType != "None")
            {
                if (model.Pass[passNumber].CalcData.ph < model.Pass[passNumber].CalcData.fph)
                    alk_ph3(model, passNumber);
                else
                    alk_ph2(model, passNumber);

                model.Pass[passNumber].CalcData.t_ion_ppm[7] = model.Pass[passNumber].CalcData.fco3 / 1.667;
                model.Pass[passNumber].CalcData.ion_ppmf[7] = model.Pass[passNumber].CalcData.t_ion_ppm[7];
                model.Pass[passNumber].CalcData.t_ion_ppm[8] = model.Pass[passNumber].CalcData.fhco3 / 0.82;
                model.Pass[passNumber].CalcData.ion_ppmf[8] = model.Pass[passNumber].CalcData.t_ion_ppm[8];
            }
            else
            {
                model.Pass[passNumber].CalcData.fco2 = 0.82 * model.Pass[passNumber].CalcData.t_ion_ppm[8] / (Math.Pow(10, model.Pass[passNumber].CalcData.fph - 6.3));
            }

            if ((model.Pass[passNumber].CalcData.ph - model.Pass[passNumber].CalcData.fph) > 0.005)
            {
                if (chemType == "H2SO4")
                {
                    model.Pass[passNumber].CalcData.t_ion_ppm[9] = model.Pass[passNumber].CalcData.t_ion_ppm[9] + model.Pass[passNumber].CalcData.fso4;
                    model.Pass[passNumber].CalcData.ion_ppmf[9] = model.Pass[passNumber].CalcData.t_ion_ppm[9];
                    model.Pass[passNumber].CalcData.dos = model.Pass[passNumber].CalcData.acids / 0.98;
                }
                if (chemType == "HCl")
                {
                    model.Pass[passNumber].CalcData.t_ion_ppm[10] = model.Pass[passNumber].CalcData.t_ion_ppm[10] + model.Pass[passNumber].CalcData.fcl;
                    model.Pass[passNumber].CalcData.ion_ppmf[10] = model.Pass[passNumber].CalcData.t_ion_ppm[10];
                    model.Pass[passNumber].CalcData.dos = model.Pass[passNumber].CalcData.acidc / 0.37;
                }
            }

            if ((model.Pass[passNumber].CalcData.fph - model.Pass[passNumber].CalcData.ph) > 0.005)
            {
                if (chemType == "NaOH")
                {
                    model.Pass[passNumber].CalcData.t_ion_ppm[2] = model.Pass[passNumber].CalcData.t_ion_ppm[2] + model.Pass[passNumber].CalcData.fna;
                    model.Pass[passNumber].CalcData.ion_ppmf[2] = model.Pass[passNumber].CalcData.t_ion_ppm[2];
                    model.Pass[passNumber].CalcData.dos = model.Pass[passNumber].CalcData.caustic / 0.3;  // 30% Caustic soda
                }
            }
        }

        private void alk_ph2(ProjectionModel model, int passNumber)
        {
            double k1 = 0.000000445;
            double k2 = 0.0000000000469;
            double h = 0;
            double ta = 0;
            double hexpr = 0;
            if (model.Pass[passNumber].CalcData.ph < 8.3)
            {
                h = Math.Pow(10, -model.Pass[passNumber].CalcData.ph);
                hexpr = (h / k1 + k2 / h + (double)1);
                ta = model.Pass[passNumber].CalcData.ion_ppm[8] / (double)61000 * hexpr;
            }
            else
            {
                ta = model.Pass[passNumber].CalcData.ion_ppm[8] / (double)61000 + model.Pass[passNumber].CalcData.ion_ppm[7] / (double)60000;
            }

            h = Math.Pow(10, -model.Pass[passNumber].CalcData.fph);
            hexpr = (h / k1 + k2 / h + (double)1);

            if (ta > 0)
            {
                model.Pass[passNumber].CalcData.fco2 = Math.Pow(10, Math.Log10(ta) - Math.Log10(hexpr) + Math.Log10(h / k1)) * (double)44000;
                model.Pass[passNumber].CalcData.fhco3 = Math.Pow(10, Math.Log10(ta) - Math.Log10(hexpr)) * (double)50000;
                model.Pass[passNumber].CalcData.fco3 = Math.Pow(10, Math.Log10(ta) - Math.Log10(hexpr) + Math.Log10(k2 / h)) * (double)100000;
                if ((model.Pass[passNumber].CalcData.ph - model.Pass[passNumber].CalcData.fph) > 0.005)
                    model.Pass[passNumber].CalcData.acids = 0.98 * (0.82 * model.Pass[passNumber].CalcData.ion_ppm[8] - model.Pass[passNumber].CalcData.fhco3 + 1.667 * model.Pass[passNumber].CalcData.ion_ppm[7] - model.Pass[passNumber].CalcData.fco3) + (0.00000000000001 / (Math.Pow(10, -1 * model.Pass[passNumber].CalcData.ph)) - 0.00000000000001 / (Math.Pow(10, -1 * model.Pass[passNumber].CalcData.fph))) * (double)50000;
            }
            else
            {
                model.Pass[passNumber].CalcData.fco2 = 0;
                model.Pass[passNumber].CalcData.fco3 = 0;
                model.Pass[passNumber].CalcData.fhco3 = 0;
                model.Pass[passNumber].CalcData.acids = 0;
            }

            if (model.Pass[passNumber].CalcData.acids < 0.01)
                model.Pass[passNumber].CalcData.acids = 0;

            model.Pass[passNumber].CalcData.acidc = 0.74 * model.Pass[passNumber].CalcData.acids;
            model.Pass[passNumber].CalcData.fso4 = 0.98 * model.Pass[passNumber].CalcData.acids;
            model.Pass[passNumber].CalcData.fcl = 0.97 * model.Pass[passNumber].CalcData.acidc;
        }

        private void alk_ph3(ProjectionModel model, int passNumber)
        {
            double k1 = 0.000000445;
            double k2 = 0.0000000000469;
            double h = 0;
            double ta = 0;
            double hexpr = 0;

            if (model.Pass[passNumber].CalcData.ph < 8.3)
            {
                h = Math.Pow(10, -model.Pass[passNumber].CalcData.ph);
                hexpr = (h / k1 + k2 / h + (double)1);
                ta = model.Pass[passNumber].CalcData.ion_ppm[8] / (double)61000 * hexpr;
            }
            else
            {
                ta = model.Pass[passNumber].CalcData.ion_ppm[8] / (double)61000 + model.Pass[passNumber].CalcData.ion_ppm[7] / (double)60000;
            }
            h = Math.Pow(10, -model.Pass[passNumber].CalcData.fph);
            hexpr = (h / k1 + k2 / h + (double)1);

            if (ta > 0)
            {
                model.Pass[passNumber].CalcData.fco2 = Math.Pow(10, Math.Log10(ta) - Math.Log10(hexpr) + Math.Log10(h / k1)) * (double)44000;
                model.Pass[passNumber].CalcData.fhco3 = Math.Pow(10, Math.Log10(ta) - Math.Log10(hexpr)) * (double)50000;
                model.Pass[passNumber].CalcData.fco3 = Math.Pow(10, Math.Log10(ta) - Math.Log10(hexpr) + Math.Log10(k2 / h)) * (double)100000;
                if ((model.Pass[passNumber].CalcData.fph - model.Pass[passNumber].CalcData.ph) > 0.005)
                    model.Pass[passNumber].CalcData.caustic = 0.8 * (model.Pass[passNumber].CalcData.fhco3 - 0.82 * model.Pass[passNumber].CalcData.ion_ppm[8] + model.Pass[passNumber].CalcData.fco3 - 1.667 * model.Pass[passNumber].CalcData.ion_ppm[7]) + (0.00000000000001 / (Math.Pow(10, -1 * model.Pass[passNumber].CalcData.ph)) - 0.00000000000001 / (Math.Pow(10, -1 * model.Pass[passNumber].CalcData.fph))) * (double)50000;
            }
            else
            {
                model.Pass[passNumber].CalcData.fco2 = 0;
                model.Pass[passNumber].CalcData.fco3 = 0;
                model.Pass[passNumber].CalcData.fhco3 = 0;
                model.Pass[passNumber].CalcData.caustic = 0;
            }

            model.Pass[passNumber].CalcData.fna = 0.46 / 0.8 * model.Pass[passNumber].CalcData.caustic; // 0.46 Na ppm to CaCO3, 0.8 NaOH ppm to CaCO3
        }

        private void TempConv(ProjectionModel model, int passNumber)
        {
            if (model.Pass[0].FeedTempUnit == "C")

                model.Pass[passNumber].CalcData.ftemp = model.Pass[0].FeedTemp;
            else
                model.Pass[passNumber].CalcData.ftemp = (model.Pass[0].FeedTemp - 32) / 1.8;

            model.Pass[passNumber].CalcData.PANumd[2] = model.Pass[passNumber].CalcData.ftemp.ToString();
        }

        public void FlowConv(ProjectionModel model, int passNumber)
        {
            model.Pass[passNumber].CalcData.prod = model.Pass[passNumber].ProductFlow;

            if (model.Pass[0].UnitType == "metric")
            {
                model.Pass[passNumber].CalcData.PProd = model.Pass[passNumber].CalcData.prod * 264.2 * 24.0;
                model.Pass[passNumber].CalcData.Recirc = model.Pass[0].RecircFlow * 264.2 * 24.0;
                model.Pass[passNumber].CalcData.Blend = model.Pass[0].BlendFlow * 264.2 * 24.0;
            }
            else
            {
                model.Pass[passNumber].CalcData.PProd = (double)model.Pass[passNumber].CalcData.prod * 1440.0;
                model.Pass[passNumber].CalcData.Recirc = model.Pass[0].RecircFlow * 1440.0;
                model.Pass[passNumber].CalcData.Blend = model.Pass[0].BlendFlow * 1440.0;
            }

            model.Pass[passNumber].CalcData.PANumd[3] = model.Pass[passNumber].CalcData.PProd.ToString();
            model.Pass[passNumber].CalcData.PANumd[17] = model.Pass[passNumber].CalcData.RecircFlow.ToString();
            model.Pass[passNumber].CalcData.PANumd[18] = model.Pass[passNumber].CalcData.Blend.ToString();
        }

        private void InitFlow(ProjectionModel model, int passNumber)
        {
            if (passNumber == 1)
                model.Pass[1].Recovery = 100.0 * model.Pass[1].ProductFlow / model.Pass[0].ProductFlow;
            model.Pass[passNumber].CalcData.Rec = model.Pass[passNumber].Recovery / 100.0;
            model.Pass[passNumber].CalcData.feed = (double)((double)model.Pass[passNumber].CalcData.PProd / (double)model.Pass[passNumber].CalcData.Rec) + (double)model.Pass[passNumber].CalcData.RecircFlow;
            model.Pass[passNumber].CalcData.feedflow = model.Pass[passNumber].CalcData.feed;
            model.Pass[passNumber].CalcData.recint = model.Pass[passNumber].CalcData.PProd / model.Pass[passNumber].CalcData.feed;

            model.Pass[passNumber].CalcData.age = model.Pass[passNumber].Age;

            model.Pass[passNumber].CalcData.sp_fact = model.Pass[passNumber].SaltPassageIncreaseFactor;
            if (model.Pass[passNumber].CalcData.sp_fact == 0)
                model.Pass[passNumber].CalcData.sp_fact = 1;

            model.Pass[passNumber].CalcData.bfact = Math.Log(model.Pass[passNumber].CalcData.sp_fact) / (double)3;
            model.Pass[passNumber].CalcData.bfact = Math.Exp(model.Pass[passNumber].CalcData.bfact * model.Pass[passNumber].CalcData.age);
            if (model.Pass[passNumber].CalcData.age == 0)
                model.Pass[passNumber].CalcData.age = (double)1 / (double)365;
        }

        private void MembranInfo(ProjectionModel model, int passNumber)
        {
            // For right now, assume that the first one in the
            // list index is always chosen

            for (int i = 0; i <= model.Pass[passNumber].CalcData.nbanks - 1; i++)
            {
                var modelNumber = GetModelNumber(model, passNumber, i);

                // Read pnetn for each element 'MRB 7 / 17 / 99                
                model.Pass[passNumber].CalcData.TCor = Math.Exp(_elements[modelNumber].TCor * ((double)1 / (273 + model.Pass[passNumber].CalcData.ftemp) - (double)1 / 298));
                model.Pass[passNumber].CalcData.pnetn[i] = _elements[modelNumber].PSTD;
                switch (_elements[modelNumber].Model)
                {
                    case "SB":
                    case "CA":
                        model.Pass[passNumber].CalcData.col[i] = _SB_Type;
                        for (int m = 0; m <= 13; m++)
                            model.Pass[passNumber].CalcData.Rej_Ions[m] = 1 - (((100 - _elements[modelNumber].Reject / 100) * model.Pass[passNumber].CalcData.Ion_Data[m].SB_Type));
                        break;
                    case "X201":
                    case "X-20":
                        model.Pass[passNumber].CalcData.col[i] = _A153_TS80_Type;
                        for (int m = 0; m <= 13; m++)
                            model.Pass[passNumber].CalcData.Rej_Ions[m] = 1 - (100 - _elements[modelNumber].Reject) / 100 * model.Pass[passNumber].CalcData.Ion_Data[m].A153_TS80_Type;
                        break;
                    case "TS80":
                    case "TS40":
                    case "XN40":
                    case "XN45":
                        model.Pass[passNumber].CalcData.col[i] = _Other_Type;

                        for (int m = 0; m <= 13; m++)
                            model.Pass[passNumber].CalcData.Rej_Ions[m] = 1 - (100 - _elements[modelNumber].Reject) / 100 * model.Pass[passNumber].CalcData.Ion_Data[m].A153_TS80_Type;
                        break;
                    case "ACM1":
                    case "ACM2":
                    case "ACM3":
                    case "ACM4":
                    case "ACMB":
                    case "ACM":
                        model.Pass[passNumber].CalcData.col[i] = _A152_ACM1_Type;
                        for (int m = 0; m <= 13; m++)
                            model.Pass[passNumber].CalcData.Rej_Ions[m] = 1 - (100 - _elements[modelNumber].Reject) / 100 * model.Pass[passNumber].CalcData.Ion_Data[m].A152_ACM1_Type;
                        break;
                    case "ACMS":
                        model.Pass[passNumber].CalcData.col[i] = _A152_ACM1_Type;
                        for (int m = 0; m <= 13; m++)
                            model.Pass[passNumber].CalcData.Rej_Ions[m] = 1 - (100 - _elements[modelNumber].Reject) / 100 * model.Pass[passNumber].CalcData.Ion_Data[m].A152_ACM1_Type;
                        break;
                    default:
                        model.Pass[passNumber].CalcData.col[i] = _Other_Type;
                        if (_elements[modelNumber].Reject == 0)
                        {
                            for (int m = 0; m <= 13; m++)
                                model.Pass[passNumber].CalcData.Rej_Ions[m] = 0;
                        }
                        else
                        {
                            for (int m = 0; m <= 13; m++)
                                model.Pass[passNumber].CalcData.Rej_Ions[m] = (double)1 - ((double)100 - _elements[modelNumber].Reject) / (double)100 * model.Pass[passNumber].CalcData.Ion_Data[m].Other_Type;
                        }
                        break;
                }

                // MH - 09/02/2019 Not sure intention here but didn't seem necessary so commented out
                //if (modelNumber > 99999)
                //{
                //    // 'This part still needs to be implemented
                //}
                //else
                //{
                model.Pass[passNumber].CalcData.el_bank[i] = model.Pass[passNumber].CalcData.El_tu[i] * model.Pass[passNumber].CalcData.Tube[i];
                model.Pass[passNumber].CalcData.area[i] = _elements[modelNumber].area;
                model.Pass[passNumber].CalcData.TotalArea = model.Pass[passNumber].CalcData.TotalArea + model.Pass[passNumber].CalcData.el_bank[i] * model.Pass[passNumber].CalcData.area[i];
                model.Pass[passNumber].CalcData.el_tot = model.Pass[passNumber].CalcData.el_tot + model.Pass[passNumber].CalcData.el_bank[i];
                model.Pass[passNumber].CalcData.leak[i] = _elements[modelNumber].leak;
                model.Pass[passNumber].CalcData.pcoef[i] = _elements[modelNumber].pcoef;
                if (_elements[modelNumber].Reject == 0)
                    model.Pass[passNumber].CalcData.rej[i] = 0;
                else
                    model.Pass[passNumber].CalcData.rej[i] = (1 - _elements[modelNumber].Reject / 100) * model.Pass[passNumber].CalcData.bfact / model.Pass[passNumber].CalcData.TCor;

                model.Pass[passNumber].CalcData.atc[i] = _elements[modelNumber].Atc1 / model.Pass[passNumber].CalcData.TCor;
                //}
                model.Pass[passNumber].CalcData.fdecl = model.Pass[passNumber].FluxDecline;
                model.Pass[passNumber].CalcData.afact[i] = model.Pass[passNumber].CalcData.atc[i] * Math.Pow(model.Pass[passNumber].CalcData.age * 365, -model.Pass[passNumber].CalcData.fdecl);
                model.Pass[passNumber].CalcData.afact[i] = model.Pass[passNumber].CalcData.afact[i] * (1 + model.Pass[passNumber].CalcData.leak[i]);

                // MH - Added this assignment since Fill Form from VB6 isn't called
                model.Pass[passNumber].CalcData.pExp[i] = _elements[modelNumber].pExp;
            }
        }

        private string GetModelNumber(ProjectionModel model, int passNumber, int index)
        {
            return model.Pass[passNumber].StageData[index].Model;
        }

        private ServiceResult InitPressure(ProjectionModel model, int passNumber)
        {
            for (int n = 0; n <= model.Pass[passNumber].CalcData.nbanks - 1; n++)
            {
                model.Pass[passNumber].CalcData.prod_sn[n] = 0.0144 * model.Pass[passNumber].CalcData.afact[n] * model.Pass[passNumber].CalcData.area[n] * model.Pass[passNumber].CalcData.pnetn[n];
                model.Pass[passNumber].CalcData.prod_snt[n] = model.Pass[passNumber].CalcData.prod_sn[n] * model.Pass[passNumber].CalcData.el_bank[n];
                model.Pass[passNumber].CalcData.prod_toti = model.Pass[passNumber].CalcData.prod_toti + model.Pass[passNumber].CalcData.prod_snt[n];
                model.Pass[passNumber].CalcData.prod_totp = model.Pass[passNumber].CalcData.prod_totp + model.Pass[passNumber].CalcData.prod_snt[n] * model.Pass[passNumber].CalcData.pnetn[n];
            }

            for (int n = 0; n <= model.Pass[passNumber].CalcData.nbanks - 1; n++)
            {
                model.Pass[passNumber].CalcData.feedav = (model.Pass[passNumber].CalcData.feed - 0.5 * model.Pass[passNumber].CalcData.prod_snt[n] * model.Pass[passNumber].CalcData.PProd / model.Pass[passNumber].CalcData.prod_toti) / ((double)1440 * model.Pass[passNumber].CalcData.Tube[n]);
                if (model.Pass[passNumber].CalcData.feedav < 0.1)
                {
                    return new ServiceResult(Result.Error, "Program does not converge. FeedAv is too low.");
                }
                var modelNumber = GetModelNumber(model, passNumber, n);
                model.Pass[passNumber].CalcData.deltapav = model.Pass[passNumber].CalcData.deltapav + model.Pass[passNumber].CalcData.pcoef[n] * Math.Pow(model.Pass[passNumber].CalcData.feedav, _elements[modelNumber].pExp) * model.Pass[passNumber].CalcData.El_tu[n];
                model.Pass[passNumber].CalcData.feed = model.Pass[passNumber].CalcData.feed - model.Pass[passNumber].CalcData.prod_snt[n] * model.Pass[passNumber].CalcData.PProd / model.Pass[passNumber].CalcData.prod_toti;
                model.Pass[passNumber].CalcData.rej[n] = model.Pass[passNumber].CalcData.rej[n] * model.Pass[passNumber].CalcData.atc[n] * model.Pass[passNumber].CalcData.TCor * model.Pass[passNumber].CalcData.pnetn[n] * model.Pass[passNumber].CalcData.area[n] * 0.00001;
            }
            model.Pass[passNumber].CalcData.pnetn_av = model.Pass[passNumber].CalcData.prod_totp / model.Pass[passNumber].CalcData.prod_toti;
            Osm_p(model.Pass[passNumber].CalcData.t_ion_ppm, model.Pass[passNumber].CalcData.ftemp, val => model.Pass[passNumber].CalcData.osm_c = val, model.Pass[passNumber].CalcData.Ion_Data);
            model.Pass[passNumber].CalcData.press_l = model.Pass[passNumber].CalcData.pnetn_av * model.Pass[passNumber].CalcData.PProd / model.Pass[passNumber].CalcData.prod_toti + model.Pass[passNumber].CalcData.osm_c;
            if (model.Pass[passNumber].CalcData.press_l < 5)
            {
                return new ServiceResult(Result.Error, "Pressure too low.");
            }
            model.Pass[passNumber].CalcData.ppress_sum = 0;
            for (int n = 0; n <= model.Pass[passNumber].CalcData.nbanks - 1; n++)
            {
                model.Pass[passNumber].CalcData.ppress_sum = model.Pass[passNumber].CalcData.Ppress[n] - model.Pass[passNumber].CalcData.BoostPress[n] + model.Pass[passNumber].CalcData.ppress_sum;
                model.Pass[passNumber].CalcData.avg_ppress = model.Pass[passNumber].CalcData.ppress_sum / model.Pass[passNumber].CalcData.nbanks;

                model.Pass[passNumber].CalcData.press_h = model.Pass[passNumber].CalcData.press_l + model.Pass[passNumber].CalcData.osm_c * (1 / ((double)1 - model.Pass[passNumber].CalcData.Rec)) + model.Pass[passNumber].CalcData.deltapav + (double)10 * model.Pass[passNumber].CalcData.nbanks + model.Pass[passNumber].CalcData.avg_ppress;
            }

            return new ServiceResult(Result.Success);
        }

        private double Osm_p(double[] ion_ppm, double temp, Action<double> osmp, List<IonDataTypeModel> ion_data)
        {

            double[] ion_moll = new double[14];
            double mollsum = 0;
            double tds = 0;

            for (int i = 0; i <= 13; i++)
            {
                tds = tds + ion_ppm[i];
            }

            for (int i = 0; i <= 12; i++)
            {
                ion_moll[i] = 0.001 * ion_ppm[i] / (ion_data[i].Weight * (1 - tds / (double)1000000));
                mollsum = mollsum + ion_moll[i];
            }

            osmp(1.12 * (temp + 273) * mollsum);

            //Debug.WriteLine("Osm_p() ion_moll(0): " + ion_moll[0]);
            //Debug.WriteLine("Osm_p() ion_moll(10): " + ion_moll[10]);
            //Debug.WriteLine("Osm_p() osm_c: " + model.CalcData.osm_c);
            //Debug.WriteLine("Osm_p() tds: " + tds);
            return tds;
        }

        private ServiceResult Calc1Pass(ProjectionModel model, int passNumber)
        {
            if (model.Pass[passNumber].CalcData.RetStatus == 0)
                Iterate(model, passNumber);

            for (int k = 0; k <= model.Pass[passNumber].CalcData.nbanks - 1; k++)
            {
                for (int j = 1; j <= model.Pass[passNumber].CalcData.El_tu[k]; j++)
                {
                    if (model.Pass[0].UnitType == "imperial")
                    {
                        if (model.Pass[passNumber].CalcData.pnetiElem[k, j] < 1)
                        {
                            return new ServiceResult(Result.Error, "Net Driving Pressure is too low. Program Does Not Converge.");
                        }
                        else if (model.Pass[passNumber].CalcData.pnetiElem[k, j] < 10)
                        {
                            model.Pass[passNumber].CalcData.PnetiWarning = 1;
                        }
                    }
                    else
                    {
                        if (model.Pass[passNumber].CalcData.pnetiElem[k, j] < 0.0689)
                        {
                            return new ServiceResult(Result.Error, "Net Driving Pressure is too low. Program Does Not Converge.");
                        }
                        else if (model.Pass[passNumber].CalcData.pnetiElem[k, j] < 0.69)
                        {
                            model.Pass[passNumber].CalcData.PnetiWarning = 1;
                        }
                    }

                    if (model.Pass[passNumber].CalcData.beta[k] > 1.2)
                        model.Pass[passNumber].CalcData.BetaWarning = 1;

                }
            }

            switch (model.Pass[passNumber].CalcData.RetStatus)
            {
                case 0:
                    if (model.Pass[passNumber].CalcData.rej[passNumber] > 0)
                    {
                        for (int i = 0; i <= 13; i++)
                            model.Pass[passNumber].CalcData.ion_ppmp[i] = (double)1440 * model.Pass[passNumber].CalcData.scp[i] / model.Pass[passNumber].CalcData.PProd;
                    }

                    for (int i = 0; i <= 13; i++)
                        model.Pass[passNumber].CalcData.ion_ppmp[i] = (model.Pass[passNumber].CalcData.PProd * model.Pass[passNumber].CalcData.ion_ppmp[i] + model.Pass[passNumber].CalcData.Blend * model.Pass[passNumber].CalcData.t_ion_ppm[i]) / (model.Pass[passNumber].CalcData.PProd + model.Pass[passNumber].CalcData.Blend);

                    int prepI = Prepare_Passes(model, passNumber);
                    if (prepI != 0)
                    {
                        model.Pass[passNumber].CalcData.RetStatus = 1;
                    }
                    else
                    {
                        model.Pass[passNumber].CalcData.Feed_Ret = model.Pass[passNumber].CalcData.fee;
                        model.Pass[passNumber].CalcData.Con_Ret = model.Pass[passNumber].CalcData.con;
                        model.Pass[passNumber].CalcData.avsp = ((double)1 - _elements[GetModelNumber(model, passNumber, 0)].Reject / (double)100);
                        Prod_Conc_ph2(model, passNumber);
                        if (model.Pass[passNumber].CalcData.cph2 == 0)
                            model.Pass[passNumber].CalcData.cph2 = model.Pass[passNumber].CalcData.fph;
                        if (model.Pass[passNumber].CalcData.pph2 == 0)
                            model.Pass[passNumber].CalcData.pph2 = model.Pass[passNumber].CalcData.fph;
                        model.Pass[passNumber].CalcData.cph = model.Pass[passNumber].CalcData.cph2;
                        model.Pass[passNumber].CalcData.pph = model.Pass[passNumber].CalcData.pph2;
                        model.Pass[passNumber].ChemDosage = model.Pass[passNumber].CalcData.dos;
                        model.Pass[passNumber].FeedCO2 = model.Pass[passNumber].CalcData.fco2;

                        model.Pass[passNumber].SystemFlux = model.Pass[passNumber].CalcData.SystemFlux;

                        // Calculate weekly and monthly dosage
                        model.Pass[passNumber].CalcData.dos_daily = (model.Pass[passNumber].CalcData.dos * model.Pass[passNumber].CalcData.PProd * 100) / (model.Pass[passNumber].Recovery * 120091);
                        model.Pass[passNumber].CalcData.dos_monthly = model.Pass[passNumber].CalcData.dos_daily * 30;
                    }
                    break;
                case 6:
                    return new ServiceResult(Result.Error, "Program Does Not Converge.");
                    break;
            }

            if (model.Pass[passNumber].CalcData.cph2 == 0)
                model.Pass[passNumber].CalcData.cph2 = model.Pass[passNumber].CalcData.fph;
            if (model.Pass[passNumber].CalcData.pph2 == 0)
                model.Pass[passNumber].CalcData.pph2 = model.Pass[passNumber].CalcData.fph;
            model.Pass[passNumber].CalcData.cph = model.Pass[passNumber].CalcData.cph2;
            model.Pass[passNumber].CalcData.pph = model.Pass[passNumber].CalcData.pph2;

            return new ServiceResult(Result.Success);
        }

        private ServiceResult CalcWithRecirc(ProjectionModel model)
        {
            double localRecircFlow = model.Pass[0].RecircFlow;  // MH - there is also a global RecircFlow in VB6 code. May cause problems.
            double RecircOriginal = model.Pass[0].RecircFlow;
            double Recovery;

            FlowConv(model, 0);
            localRecircFlow = model.Pass[0].CalcData.Recirc;
            model.Pass[0].CalcData.RecircFlowOriginal = localRecircFlow;
            Recovery = model.Pass[0].Recovery / 100;
            model.Pass[0].RecircFlow = 0;

            DebugOut(model, 0);

            var serviceResult = Initialize(model, 0, 0);

            DebugOut(model, 0);

            serviceResult = Calc1Pass(model, 0);
            if (serviceResult.Result != Result.Success)
                return serviceResult;

            DebugOut(model, 0);
            model.Pass[0].Recovery = model.Pass[0].CalcData.PProd / ((model.Pass[0].CalcData.PProd / Recovery) + localRecircFlow);
            model.Pass[0].Recovery = model.Pass[0].Recovery * 100;

            for (int i = 0; i <= 13; i++)
                model.Pass[0].CalcData.t_ion_ppm[i] = (model.Pass[0].CalcData.t_ion_ppm[i] * model.Pass[0].CalcData.PProd / Recovery + model.Pass[0].CalcData.ion_ppmc[i] * localRecircFlow) / (localRecircFlow + model.Pass[0].CalcData.PProd / Recovery);

            serviceResult = Initialize(model, 0, 1);
            if (serviceResult.Result != Result.Success)
                return serviceResult;

            serviceResult = Calc1Pass(model, 0);
            if (serviceResult.Result != Result.Success)
                return serviceResult;

            model.Pass[0].Recovery = Recovery * 100;
            model.Pass[0].RecircFlow = RecircOriginal;

            model.bCalc = true;

            return new ServiceResult(Result.Success);
        }

        private ServiceResult Calc2Pass(ProjectionModel model)
        {
            bool bCont = true;
            double tds = 0;
            double tds2 = 1;
            model.Pass[0].RecircFlow = 0;

            var serviceResult = Initialize(model, 0, 0);
            if (serviceResult.Result != Result.Success)
                return serviceResult;

            serviceResult = Calc1Pass(model, 0);
            if (serviceResult.Result != Result.Success)
                return serviceResult;

            serviceResult = Initialize(model, 1, 2);
            if (serviceResult.Result != Result.Success)
                return serviceResult;

            serviceResult = Calc1Pass(model, 1);
            if (serviceResult.Result != Result.Success)
                return serviceResult;

            while (bCont)
            {
                serviceResult = Initialize(model, 0, 3);
                if (serviceResult.Result != Result.Success)
                    return serviceResult;

                serviceResult = Calc1Pass(model, 0);
                if (serviceResult.Result != Result.Success)
                    return serviceResult;

                tds = 0;
                for (int k = 0; k <= 13; k++)
                    tds = tds + model.Pass[0].CalcData.ion_ppmf[k];

                if (Math.Abs(tds - tds2) / tds2 < 0.02)
                {
                    bCont = false;
                    serviceResult = Initialize(model, 1, 2);
                    if (serviceResult.Result != Result.Success)
                        return serviceResult;

                    serviceResult = Calc1Pass(model, 1);
                    if (serviceResult.Result != Result.Success)
                        return serviceResult;

                    serviceResult = Initialize(model, 0, 3);
                    if (serviceResult.Result != Result.Success)
                        return serviceResult;

                    serviceResult = Calc1Pass(model, 0);
                    if (serviceResult.Result != Result.Success)
                        return serviceResult;

                }
                if (bCont)
                {
                    tds2 = tds;
                    serviceResult = Initialize(model, 1, 2);
                    if (serviceResult.Result != Result.Success)
                        return serviceResult;

                    serviceResult = Calc1Pass(model, 1);
                    if (serviceResult.Result != Result.Success)
                        return serviceResult;
                }
            }

            model.Pass[0].RecircFlow = model.Pass[0].ProductFlow - model.Pass[1].ProductFlow;

            return new ServiceResult(Result.Success);
        }

        private void Iterate(ProjectionModel model, int passNumber)
        {
            bool inter;  // flag to break out of nested loops
            int n;
            double dConvPress;
            double dConvFlow;
            double dConvFlux;

            // MH - Debug
            Debug.WriteLine("ITERATE()");

            if (model.Pass[0].UnitType == "imperial")
            {
                dConvPress = 1;
                dConvFlow = 1;
                dConvFlux = 1;
            }
            else
            {
                dConvPress = _PSItoBAR;
                dConvFlow = _GPMtoM3;
                dConvFlux = _GFDtoLMH;
            }

            for (n = 1; n <= 42; n++)
            {
                inter = false;
                model.Pass[passNumber].CalcData.flow_f = model.Pass[passNumber].CalcData.PProd / ((double)1440 * model.Pass[passNumber].CalcData.Rec);
                model.Pass[passNumber].CalcData.flow_pt = model.Pass[passNumber].CalcData.PProd / ((double)1440 * model.Pass[passNumber].CalcData.el_tot);
                model.Pass[passNumber].CalcData.pressf = (model.Pass[passNumber].CalcData.press_l + model.Pass[passNumber].CalcData.press_h) / (double)2;
                model.Pass[passNumber].CalcData.press_feed[0] = model.Pass[passNumber].CalcData.pressf;
                model.Pass[passNumber].CalcData.flow_feed[0] = model.Pass[passNumber].CalcData.flow_f;
                model.Pass[passNumber].CalcData.p_prod = 0;
                model.Pass[passNumber].CalcData.p_salp = 0;


                for (int i = 0; i <= 13; i++)
                {
                    model.Pass[passNumber].CalcData.ion_ppmp[i] = (1 - model.Pass[passNumber].CalcData.Rej_Ions[i]) * model.Pass[passNumber].CalcData.t_ion_ppm[i] * (Math.Log(1 / (1 - model.Pass[passNumber].CalcData.recint)) / model.Pass[passNumber].CalcData.recint);
                    model.Pass[passNumber].CalcData.ion_ppmb[i] = ((model.Pass[passNumber].CalcData.t_ion_ppm[i] * (model.Pass[passNumber].CalcData.feedflow)) - (model.Pass[passNumber].CalcData.ion_ppmp[i] * model.Pass[passNumber].CalcData.PProd)) / (model.Pass[passNumber].CalcData.feedflow - model.Pass[passNumber].CalcData.PProd);
                    model.Pass[passNumber].CalcData.ion_ppmc[i] = model.Pass[passNumber].CalcData.t_ion_ppm[i];
                    model.Pass[passNumber].CalcData.scp[i] = (double)0;
                    model.Pass[passNumber].CalcData.scpk[i] = (double)0;
                }

                // MH - Debug
                //Debug.WriteLine("n: " + n);
                //Debug.WriteLine("ion_ppmp(0): " + model.CalcData.ion_ppmp[0]);
                //Debug.WriteLine("ion_ppmb(0): " + model.CalcData.ion_ppmb[0]);
                //Debug.WriteLine("ion_ppmc(0): " + model.CalcData.ion_ppmc[0]);
                //Debug.WriteLine("ion_ppmp(10): " + model.CalcData.ion_ppmp[10]);
                //Debug.WriteLine("ion_ppmb(10): " + model.CalcData.ion_ppmb[10]);
                //Debug.WriteLine("ion_ppmc(10): " + model.CalcData.ion_ppmc[10]);

                Osm_p(model.Pass[passNumber].CalcData.ion_ppmc, model.Pass[passNumber].CalcData.ftemp, val => model.Pass[passNumber].CalcData.osm_c = val, model.Pass[passNumber].CalcData.Ion_Data);

                Debug.WriteLine("osm_c: " + model.Pass[passNumber].CalcData.osm_c);

                model.Pass[passNumber].CalcData.BetaWarning = 0;
                model.Pass[passNumber].CalcData.PnetiWarning = 0;

                // Loop for each bank
                for (int k = 0; k <= model.Pass[passNumber].CalcData.nbanks - 1; k++)
                {
                    model.Pass[passNumber].CalcData.b_salp[k] = 0;
                    model.Pass[passNumber].CalcData.sumfp = 0;
                    model.Pass[passNumber].CalcData.flow_ft = model.Pass[passNumber].CalcData.flow_f / model.Pass[passNumber].CalcData.Tube[k];
                    model.Pass[passNumber].CalcData.rec_l = (double)0;
                    model.Pass[passNumber].CalcData.rec_h = (double)2 * model.Pass[passNumber].CalcData.Rec / (model.Pass[passNumber].CalcData.nbanks * model.Pass[passNumber].CalcData.El_tu[0]);

                    // MH - Debug
                    Debug.WriteLine("k: " + k);
                    Debug.WriteLine("flow_ft: " + model.Pass[passNumber].CalcData.flow_ft);
                    Debug.WriteLine("rec_h : " + model.Pass[passNumber].CalcData.rec_h);


                    // Loop for each tube
                    for (int j = 1; j <= model.Pass[passNumber].CalcData.El_tu[k]; j++)
                    {
                        if (!Iterate_While(model, passNumber, n, k))
                        {
                            if (model.Pass[passNumber].CalcData.RetStatus != 0)
                                return;
                            inter = true;
                            break;
                        }
                        if (inter)
                            break;

                        // Store individual element betas, flows, and tds here
                        model.Pass[passNumber].CalcData.pnetiElem[k, j] = dConvPress * model.Pass[passNumber].CalcData.pneti;
                        model.Pass[passNumber].CalcData.betaElem[k, j] = model.Pass[passNumber].CalcData.beta[k];
                        model.Pass[passNumber].CalcData.flowElem[k, j] = dConvFlow * model.Pass[passNumber].CalcData.flow_pt;
                        model.Pass[passNumber].CalcData.fluxElem[k, j] = dConvFlux * model.Pass[passNumber].CalcData.flow_pt * (double)1440 / model.Pass[passNumber].CalcData.area[k];

                        // MH - Debug
                        Debug.WriteLine("j: " + j);
                        Debug.WriteLine("pnetiElem[passNumber, k, j]: " + model.Pass[passNumber].CalcData.pnetiElem[k, j]);
                        Debug.WriteLine("betaElem[passNumber, k, j] : " + model.Pass[passNumber].CalcData.betaElem[k, j]);
                        Debug.WriteLine("flowElem[passNumber, k, j]: " + model.Pass[passNumber].CalcData.flowElem[k, j]);
                        Debug.WriteLine("fluxElem[passNumber, k, j] : " + model.Pass[passNumber].CalcData.fluxElem[k, j]);

                        for (int i = 0; i <= 13; i++)
                            model.Pass[passNumber].CalcData.tdsElem[k, j] = 0;

                        for (int i = 0; i <= 13; i++)
                            model.Pass[passNumber].CalcData.tdsElem[k, j] = model.Pass[passNumber].CalcData.tdsElem[k, j] + model.Pass[passNumber].CalcData.ion_ppmp[i];
                        // End individ

                        model.Pass[passNumber].CalcData.flow_ft = model.Pass[passNumber].CalcData.flow_ft - model.Pass[passNumber].CalcData.flow_pt;
                        model.Pass[passNumber].CalcData.sumfp = model.Pass[passNumber].CalcData.sumfp + model.Pass[passNumber].CalcData.flow_pt * model.Pass[passNumber].CalcData.Tube[k];

                        for (int i = 0; i <= 13; i++)
                            model.Pass[passNumber].CalcData.scpk[i] = model.Pass[passNumber].CalcData.scpk[i] + model.Pass[passNumber].CalcData.ion_ppmp[i] * model.Pass[passNumber].CalcData.flow_pt * model.Pass[passNumber].CalcData.Tube[k];

                        model.Pass[passNumber].CalcData.pressf = model.Pass[passNumber].CalcData.pressf - model.Pass[passNumber].CalcData.deltap;


                    }
                    // End loop for each tube

                    model.Pass[passNumber].CalcData.press_feed[k + 1] = model.Pass[passNumber].CalcData.pressf;
                    model.Pass[passNumber].CalcData.pressf = model.Pass[passNumber].CalcData.pressf - (double)3;
                    model.Pass[passNumber].CalcData.b_prod[k] = model.Pass[passNumber].CalcData.sumfp;
                    model.Pass[passNumber].CalcData.p_prod = model.Pass[passNumber].CalcData.p_prod + model.Pass[passNumber].CalcData.b_prod[k];
                    model.Pass[passNumber].CalcData.flow_f = model.Pass[passNumber].CalcData.flow_ft * model.Pass[passNumber].CalcData.Tube[k];
                    model.Pass[passNumber].CalcData.flow_feed[k + 1] = model.Pass[passNumber].CalcData.flow_f;
                    for (int i = 0; i <= 14; i++)
                    {
                        model.Pass[passNumber].CalcData.scp[i] = model.Pass[passNumber].CalcData.scp[i] + model.Pass[passNumber].CalcData.scpk[i];
                        model.Pass[passNumber].CalcData.b_salp[k] = model.Pass[passNumber].CalcData.b_salp[k] + model.Pass[passNumber].CalcData.scpk[i];
                        model.Pass[passNumber].CalcData.scpk[i] = (double)0;
                    }
                    Con_bal(model, passNumber);
                }
                if (inter)
                    continue;

                if (((Math.Abs(model.Pass[passNumber].CalcData.PProd - (double)1440 * model.Pass[passNumber].CalcData.p_prod) / model.Pass[passNumber].CalcData.PProd) <= 0.005) && (model.Pass[passNumber].CalcData.flow_f > (double)0))
                {
                    model.Pass[passNumber].CalcData.flow_feed[0] = model.Pass[passNumber].CalcData.PProd / ((double)1440 * model.Pass[passNumber].CalcData.Rec);
                    for (int k = 0; k <= model.Pass[passNumber].CalcData.nbanks - 1; k++)
                    {
                        model.Pass[passNumber].CalcData.b_prod[k] = model.Pass[passNumber].CalcData.b_prod[k] * model.Pass[passNumber].CalcData.PProd / ((double)1440 * model.Pass[passNumber].CalcData.p_prod);
                        model.Pass[passNumber].CalcData.flow_feed[k + 1] = model.Pass[passNumber].CalcData.flow_feed[k] - model.Pass[passNumber].CalcData.b_prod[k];
                    }
                    break;
                }

                if (((model.Pass[passNumber].CalcData.press_h - model.Pass[passNumber].CalcData.press_l) / model.Pass[passNumber].CalcData.press_h) < 0.001)
                {
                    model.Pass[passNumber].CalcData.press_h = 1.2 * model.Pass[passNumber].CalcData.press_h;
                    model.Pass[passNumber].CalcData.press_l = 0.8 * model.Pass[passNumber].CalcData.press_l;
                }

                if (model.Pass[passNumber].CalcData.PProd / (double)1440 < model.Pass[passNumber].CalcData.p_prod)
                    model.Pass[passNumber].CalcData.press_h = 0.5 * (model.Pass[passNumber].CalcData.press_h + model.Pass[passNumber].CalcData.press_l);
                else
                    model.Pass[passNumber].CalcData.press_l = 0.5 * (model.Pass[passNumber].CalcData.press_h + model.Pass[passNumber].CalcData.press_l);
            }

            if (n > 42)
                model.Pass[passNumber].CalcData.RetStatus = 7;
        }

        private bool Iterate_While(ProjectionModel model, int passNumber, int n, int k)
        {
            bool result = true;
            bool cont = true;
            DateTime tStart = DateTime.Now;
            double FlowFeed1;
            double Flow_ft1;

            // MH - Debug
            Debug.WriteLine("ITERATE_WHILE()");

            while (cont)
            {
                model.Pass[passNumber].CalcData.rec_av = (model.Pass[passNumber].CalcData.rec_l + model.Pass[passNumber].CalcData.rec_h) / (double)2;
                model.Pass[passNumber].CalcData.flow_b = model.Pass[passNumber].CalcData.flow_ft - model.Pass[passNumber].CalcData.flow_pt;
                model.Pass[passNumber].CalcData.flow_bav = (model.Pass[passNumber].CalcData.flow_ft + model.Pass[passNumber].CalcData.flow_b) / (double)2;

                Debug.WriteLine("rec_av: " + model.Pass[passNumber].CalcData.rec_av);
                Debug.WriteLine("flow_b: " + model.Pass[passNumber].CalcData.flow_b);
                Debug.WriteLine("flow_bav : " + model.Pass[passNumber].CalcData.flow_bav);

                if (model.Pass[passNumber].CalcData.flow_bav < 0.1)
                {
                    model.Pass[passNumber].CalcData.press_l = 0.95 * model.Pass[passNumber].CalcData.press_l;
                    model.Pass[passNumber].CalcData.press_h = 0.5 * (model.Pass[passNumber].CalcData.press_h + model.Pass[passNumber].CalcData.press_l);
                    if (model.Pass[passNumber].CalcData.press_l < 3)
                        model.Pass[passNumber].CalcData.RetStatus = 5;
                    result = false;
                }

                model.Pass[passNumber].CalcData.deltap = model.Pass[passNumber].CalcData.pcoef[k] * Math.Pow(model.Pass[passNumber].CalcData.flow_bav, model.Pass[passNumber].CalcData.pExp[k]);
                model.Pass[passNumber].CalcData.pneti = model.Pass[passNumber].CalcData.pressf - 0.5 * model.Pass[passNumber].CalcData.deltap - model.Pass[passNumber].CalcData.osm_c * ((double)1 + (double)1 / ((double)1 - model.Pass[passNumber].CalcData.rec_av)) / (double)2 - model.Pass[passNumber].CalcData.Ppress[k] + model.Pass[passNumber].CalcData.BoostPress[k];
                if (model.Pass[passNumber].CalcData.pneti < 1)
                    model.Pass[passNumber].CalcData.pneti = 0.1;

                model.Pass[passNumber].CalcData.flow_pt = model.Pass[passNumber].CalcData.afact[k] * model.Pass[passNumber].CalcData.pneti * model.Pass[passNumber].CalcData.area[k] * 0.00001;

                if ((double)2 * Math.Abs(model.Pass[passNumber].CalcData.rec_av - model.Pass[passNumber].CalcData.flow_pt / model.Pass[passNumber].CalcData.flow_ft) / (model.Pass[passNumber].CalcData.flow_pt / model.Pass[passNumber].CalcData.flow_ft + model.Pass[passNumber].CalcData.rec_av) > 0.05)
                {
                    if ((model.Pass[passNumber].CalcData.flow_pt / model.Pass[passNumber].CalcData.flow_ft) > model.Pass[passNumber].CalcData.rec_av)
                    {
                        model.Pass[passNumber].CalcData.rec_l = model.Pass[passNumber].CalcData.rec_av;
                        model.Pass[passNumber].CalcData.rec_h = model.Pass[passNumber].CalcData.flow_pt / model.Pass[passNumber].CalcData.flow_ft;
                    }
                    else
                    {
                        model.Pass[passNumber].CalcData.rec_h = model.Pass[passNumber].CalcData.rec_av;
                        model.Pass[passNumber].CalcData.rec_l = model.Pass[passNumber].CalcData.flow_pt / model.Pass[passNumber].CalcData.flow_ft;
                    }
                }
                else
                {
                    if (_elements[GetModelNumber(model, passNumber, k)].Model.Substring(1, 1) == "6")
                        model.Pass[passNumber].CalcData.beta[k] = Math.Exp(_bexp60 * model.Pass[passNumber].CalcData.flow_pt / model.Pass[passNumber].CalcData.flow_bav);
                    else
                        model.Pass[passNumber].CalcData.beta[k] = Math.Exp(_bexp40 * model.Pass[passNumber].CalcData.flow_pt / model.Pass[passNumber].CalcData.flow_bav);

                    if (model.Pass[passNumber].CalcData.col[k] == _A153_TS80_Type)
                    {
                        model.Pass[passNumber].CalcData.tdsc = Osm_p(model.Pass[passNumber].CalcData.ion_ppmc, model.Pass[passNumber].CalcData.ftemp, val => model.Pass[passNumber].CalcData.osmb = val, model.Pass[passNumber].CalcData.Ion_Data);
                        model.Pass[passNumber].CalcData.tdsf = Osm_p(model.Pass[passNumber].CalcData.t_ion_ppm, model.Pass[passNumber].CalcData.ftemp, val => model.Pass[passNumber].CalcData.osma = val, model.Pass[passNumber].CalcData.Ion_Data);
                    }
                    for (int i = 0; i <= 13; i++)
                    {
                        switch (model.Pass[passNumber].CalcData.col[k])
                        {
                            case _SB_Type:
                                model.Pass[passNumber].CalcData.ion_ppmp[i] = model.Pass[passNumber].CalcData.ion_ppmc[i] * (1 - Math.Pow(1 - model.Pass[passNumber].CalcData.rec_av, model.Pass[passNumber].CalcData.Ion_Data[i].SB_Type * model.Pass[passNumber].CalcData.rej[k])) / (model.Pass[passNumber].CalcData.flow_pt * model.Pass[passNumber].CalcData.rec_av);
                                break;
                            case _A153_TS80_Type:
                                model.Pass[passNumber].CalcData.ion_ppmp[i] = model.Pass[passNumber].CalcData.ion_ppmc[i] * (1 - Math.Pow(1 - model.Pass[passNumber].CalcData.rec_av, model.Pass[passNumber].CalcData.Ion_Data[i].A153_TS80_Type * model.Pass[passNumber].CalcData.rej[k])) / (model.Pass[passNumber].CalcData.flow_pt * model.Pass[passNumber].CalcData.rec_av);
                                break;
                            case _A152_ACM1_Type:
                                model.Pass[passNumber].CalcData.ion_ppmp[i] = model.Pass[passNumber].CalcData.ion_ppmc[i] * (1 - Math.Pow(1 - model.Pass[passNumber].CalcData.rec_av, model.Pass[passNumber].CalcData.Ion_Data[i].A152_ACM1_Type * model.Pass[passNumber].CalcData.rej[k])) / (model.Pass[passNumber].CalcData.flow_pt * model.Pass[passNumber].CalcData.rec_av);
                                break;
                            default:
                                model.Pass[passNumber].CalcData.ion_ppmp[i] = model.Pass[passNumber].CalcData.ion_ppmc[i] * (1 - Math.Pow(1 - model.Pass[passNumber].CalcData.rec_av, model.Pass[passNumber].CalcData.Ion_Data[i].Other_Type * model.Pass[passNumber].CalcData.rej[k])) / (model.Pass[passNumber].CalcData.flow_pt * model.Pass[passNumber].CalcData.rec_av);
                                break;
                        }
                        FlowFeed1 = model.Pass[passNumber].CalcData.flow_feed[0];
                        Flow_ft1 = model.Pass[passNumber].CalcData.flow_ft;
                        model.Pass[passNumber].CalcData.ion_ppmc[i] = (model.Pass[passNumber].CalcData.t_ion_ppm[i] * FlowFeed1 - model.Pass[passNumber].CalcData.scp[i] - model.Pass[passNumber].CalcData.scpk[i] - model.Pass[passNumber].CalcData.ion_ppmp[i] * model.Pass[passNumber].CalcData.flow_pt * model.Pass[passNumber].CalcData.Tube[k]) / ((Flow_ft1 - model.Pass[passNumber].CalcData.flow_pt) * model.Pass[passNumber].CalcData.Tube[k]);
                    }

                    Osm_p(model.Pass[passNumber].CalcData.ion_ppmc, model.Pass[passNumber].CalcData.ftemp, val => model.Pass[passNumber].CalcData.osm_c = val, model.Pass[passNumber].CalcData.Ion_Data);

                    model.Pass[passNumber].CalcData.rec_l = 0.5 * model.Pass[passNumber].CalcData.flow_pt / model.Pass[passNumber].CalcData.flow_ft;
                    model.Pass[passNumber].CalcData.rec_h = 1.5 * model.Pass[passNumber].CalcData.flow_pt / model.Pass[passNumber].CalcData.flow_ft;
                    cont = false;
                }
                if ((DateTime.Now - tStart).TotalSeconds > _settingsService.Settings.IterationTimeoutSeconds)
                {
                    cont = false;
                    result = false;
                    model.Pass[passNumber].CalcData.RetStatus = 6;
                }
            }
            return result;
        }

        private int Prepare_Passes(ProjectionModel model, int passNumber)
        {
            int result = 0;
            string str = "";
            double row0 = 0;
            double row1 = 0;
            double row2 = 0;
            double row3 = 0;
            double row4 = 0;
            double row5 = 0;
            double row6 = 0;
            double row8 = 0;

            string S;
            int sIndex;
            double dConvPress;
            double dConvFlow;
            double dConvFlux;

            if (model.Pass[0].UnitType == "imperial")
            {
                dConvPress = 1;
                dConvFlow = 1;
                dConvFlux = 1;
            }
            else
            {
                dConvPress = _PSItoBAR;
                dConvFlow = _GPMtoM3;
                dConvFlux = _GFDtoLMH;
            }

            for (int i = 0; i <= 13; i++)
                row8 = row8 + (double)1440 * model.Pass[passNumber].CalcData.scp[i] / model.Pass[passNumber].CalcData.PProd;

            model.Pass[passNumber].CalcData.SystemFlux = model.Pass[passNumber].CalcData.PProd * dConvFlux / model.Pass[passNumber].CalcData.TotalArea;
            sIndex = passNumber * 28;
            for (int k = 0; k <= model.Pass[passNumber].CalcData.nbanks - 1; k++)
            {
                row0 = model.Pass[passNumber].CalcData.b_prod[k] / (model.Pass[passNumber].CalcData.el_bank[k] * _elements[GetModelNumber(model, passNumber, k)].area) * (double)1440;

                model.Pass[passNumber].StageData[k].AverageFlux = dConvFlux * row0;
                if (row0 < (double)0)
                    return 9;

                if (k < 2)
                    row1 = model.Pass[passNumber].CalcData.press_feed[k] + model.Pass[passNumber].CalcData.BoostPress[k];
                else if (k == 2)
                    row1 = model.Pass[passNumber].CalcData.press_feed[k] + model.Pass[passNumber].CalcData.BoostPress[k] + model.Pass[passNumber].CalcData.BoostPress[k - 1];
                else if (k == 3)
                    row1 = model.Pass[passNumber].CalcData.press_feed[k] + model.Pass[passNumber].CalcData.BoostPress[k] + model.Pass[passNumber].CalcData.BoostPress[k - 1] + model.Pass[passNumber].CalcData.BoostPress[k - 2];

                if (k == 0)
                    model.Pass[passNumber].CalcData.fee = row1;

                model.Pass[passNumber].StageData[k].FeedPressure = dConvPress * row1;
                if (row1 < (double)0)
                    return 9;

                if (k < 2)
                    row2 = model.Pass[passNumber].CalcData.press_feed[k + 1] + model.Pass[passNumber].CalcData.BoostPress[k];
                else if (k == 2)
                    row2 = model.Pass[passNumber].CalcData.press_feed[k + 1] + model.Pass[passNumber].CalcData.BoostPress[k] + model.Pass[passNumber].CalcData.BoostPress[k - 1];
                else if (k == 3)
                    row2 = model.Pass[passNumber].CalcData.press_feed[k + 1] + model.Pass[passNumber].CalcData.BoostPress[k] + model.Pass[passNumber].CalcData.BoostPress[k - 1] + model.Pass[passNumber].CalcData.BoostPress[k - 2];

                model.Pass[passNumber].StageData[k].ConcentratedPressure = dConvPress * row2;
                model.Pass[passNumber].CalcData.con = row2;

                if (row2 < (double)0)
                    result = 9;

                row3 = model.Pass[passNumber].CalcData.flow_feed[k];
                model.Pass[passNumber].StageData[k].FeedFlow = dConvFlow * row3;

                if (row3 < (double)0)
                    return 9;


                row4 = model.Pass[passNumber].CalcData.flow_feed[k + 1];
                model.Pass[passNumber].StageData[k].ConcentratedFlow = dConvFlow * row4;

                if (row4 < (double)0)
                    return 9;

                row5 = model.Pass[passNumber].CalcData.beta[k];
                model.Pass[passNumber].StageData[k].BetaLastElement = row5;

                if (row5 < (double)0)
                    result = 9;

                row6 = model.Pass[passNumber].CalcData.b_salp[k] / model.Pass[passNumber].CalcData.b_prod[k];

                model.Pass[passNumber].StageData[k].ProductSalinity = row6;
                if (row6 < 0)
                    return 9;
            }
            return result;
        }

        private void Prod_Conc_ph2(ProjectionModel model, int passNumber)
        {
            if (model.Pass[passNumber].CalcData.ion_ppmc[8] > 0 && model.Pass[passNumber].CalcData.scp[8] > 0 && model.Pass[passNumber].CalcData.fco2 > 0)
            {
                if ((model.Pass[passNumber].CalcData.fph <= 8.3) || (model.Pass[passNumber].CalcData.ion_ppmc[7] / model.Pass[passNumber].CalcData.ion_ppmc[8] < 0.015))
                {
                    model.Pass[passNumber].CalcData.pph2 = 6.3 + Math.Log10((0.82 * (double)1440 * model.Pass[passNumber].CalcData.scp[8] / model.Pass[passNumber].CalcData.PProd) / model.Pass[passNumber].CalcData.fco2);
                    model.Pass[passNumber].CalcData.cph2 = 6.3 + Math.Log10(0.82 * model.Pass[passNumber].CalcData.ion_ppmc[8] / model.Pass[passNumber].CalcData.fco2);
                }
                else if (model.Pass[passNumber].CalcData.fph > 8.3 && model.Pass[passNumber].CalcData.ion_ppmc[7] > 0)
                {
                    model.Pass[passNumber].CalcData.pph2 = 6.3 + Math.Log10((0.82 * (double)1440 * model.Pass[passNumber].CalcData.scp[8] / model.Pass[passNumber].CalcData.PProd) / model.Pass[passNumber].CalcData.fco2);
                    model.Pass[passNumber].CalcData.cph2 = 9.96 + Math.Log10((1.67 * model.Pass[passNumber].CalcData.ion_ppmc[7]) / (0.82 * model.Pass[passNumber].CalcData.ion_ppmc[8]));
                }
                else
                {
                    model.Pass[passNumber].CalcData.pph2 = 0;
                    model.Pass[passNumber].CalcData.cph2 = 0;
                }
            }
            else
            {
                model.Pass[passNumber].CalcData.pph2 = 0;
                model.Pass[passNumber].CalcData.cph2 = 0;
            }
        }

        private void Con_bal(ProjectionModel model, int passNumber)
        {
            double tot_pos = 0;
            double tot_neg = 0;
            double[] ion_eqv = new double[14];

            for (int i = 0; i <= 13; i++)
            {
                ion_eqv[i] = Math.Abs(model.Pass[passNumber].CalcData.scp[i] * model.Pass[passNumber].CalcData.Ion_Data[i].Valency / model.Pass[passNumber].CalcData.Ion_Data[i].Weight);
                if (model.Pass[passNumber].CalcData.Ion_Data[i].Valency > 0)
                    tot_pos = tot_pos + ion_eqv[i];
                else
                    tot_neg = tot_neg + ion_eqv[i];
            }

            if (Math.Abs(2 * (tot_pos - tot_neg)) / (tot_pos + tot_neg) > 0.02)
            {
                for (int i = 0; i <= 6; i++)
                {
                    model.Pass[passNumber].CalcData.scp[i] = model.Pass[passNumber].CalcData.scp[i] - 0.5 * (tot_pos - tot_neg) * ion_eqv[i] / tot_pos * model.Pass[passNumber].CalcData.Ion_Data[i].Weight / model.Pass[passNumber].CalcData.Ion_Data[i].Valency;
                    if (model.Pass[passNumber].CalcData.scp[i] < (double)0)
                        model.Pass[passNumber].CalcData.scp[i] = (double)0;

                    if (model.Pass[passNumber].CalcData.scp[i] > model.Pass[passNumber].CalcData.t_ion_ppm[i] * model.Pass[passNumber].CalcData.p_prod)
                        model.Pass[passNumber].CalcData.scp[i] = model.Pass[passNumber].CalcData.t_ion_ppm[i] * model.Pass[passNumber].CalcData.p_prod;

                    model.Pass[passNumber].CalcData.ion_ppmc[i] = (model.Pass[passNumber].CalcData.t_ion_ppm[i] * (model.Pass[passNumber].CalcData.p_prod + model.Pass[passNumber].CalcData.flow_f) - model.Pass[passNumber].CalcData.scp[i]) / model.Pass[passNumber].CalcData.flow_f;
                }
                for (int i = 7; i <= 13; i++)
                {
                    model.Pass[passNumber].CalcData.scp[i] = model.Pass[passNumber].CalcData.scp[i] - 0.5 * (tot_pos - tot_neg) * ion_eqv[i] / tot_neg * model.Pass[passNumber].CalcData.Ion_Data[i].Weight / model.Pass[passNumber].CalcData.Ion_Data[i].Valency;
                    if (model.Pass[passNumber].CalcData.scp[i] < (double)0)
                        model.Pass[passNumber].CalcData.scp[i] = (double)0;

                    if (model.Pass[passNumber].CalcData.scp[i] > model.Pass[passNumber].CalcData.t_ion_ppm[i] * model.Pass[passNumber].CalcData.p_prod)
                        model.Pass[passNumber].CalcData.scp[i] = model.Pass[passNumber].CalcData.t_ion_ppm[i] * model.Pass[passNumber].CalcData.p_prod;

                    model.Pass[passNumber].CalcData.ion_ppmc[i] = (model.Pass[passNumber].CalcData.t_ion_ppm[i] * (model.Pass[passNumber].CalcData.p_prod + model.Pass[passNumber].CalcData.flow_f) - model.Pass[passNumber].CalcData.scp[i]) / model.Pass[passNumber].CalcData.flow_f;
                }
            }
        }

        private void Dosage(ProjectionModel model, int passNumber, ref string S, ref string t, ref string u, ref string v, double Satindex, double CaSulfateSat,
            double CaFluorideSat, double StronSulfateSat, double BariumSulfateSat, double SilicaSat, ref string sScalant, ref double d1, ref double d2,
            ref double d3, ref double d4, ref double d5)
        {
            string sTripol;
            double Base_Dose;
            double ppmSi;
            double Inj_Rate;
            double Solution;
            double Mix = 0.0;
            double Water = 0.0;
            double Daily_Usage;
            double Month_Usage;
            string[] Messages = new string[20];
            bool[] Statuses = new bool[20];
            bool Warning = false;
            double FeedFlowforDose;
            double RecircDoseFlow;
            double RecDose;

            Messages[0] = "*** WARNING *** CaCO3 solubility exceeded, add scale inhibitor";
            Messages[1] = "*** WARNING *** Maximum LSI exceeded, lower feed pH or decrease recovery.";
            Messages[2] = "*** WARNING *** CaSO4 solubility exceeded, add scale inhibitor";
            Messages[3] = "*** WARNING *** Maximum CaSO4 saturation exceeded, decrease recovery.";
            Messages[4] = "*** WARNING *** SrSO4 solubility exceeded, add scale inhibitor";
            Messages[5] = "*** WARNING *** Maximum SrSO4 saturation exceeded, decrease recovery.";
            Messages[6] = "*** WARNING *** BasSO4 solubility exceeded, add scale inhibitor";
            Messages[7] = "*** WARNING *** Maximum BaSO4 saturation exceeded, decrease recovery.";
            Messages[8] = "*** WARNING *** CaF2 solubility exceeded, add scale inhibitor";
            Messages[9] = "*** WARNING *** Maximum CaF2 saturation exceeded, decrease recovery.";
            Messages[10] = "*** WARNING *** Maximum Silica concentration exceeded, decrease recovery.";
            Messages[11] = "*** WARNING *** Silica solubility exceeded, add scale inhibitor";
            Messages[12] = "*** WARNING *** Maximum Beta Value of 1.2 exceeded. Reduce recovery or increase element path length!";
            Messages[13] = "*** WARNING *** Minimum Net Driving Pressure is Below 10 psi. Adjust Recovery or Feed Pressure!";
            Messages[14] = "*** WARNING *** Minimum Net Driving Pressure is Below 0.7 Bar. Adjust Recovery or Feed Pressure!";
            Messages[15] = "*** WARNING *** Iron should be limited to 0.05 mg/L to prevent iron fouling. Please refer to our Pretreatment Guides for more information.";
            Messages[16] = "*** WARNING *** Manganese should be limited to 0.05 mg/L to prevent manganese fouling. Please refer to our Pretreatment Guides for more information.";
            Messages[17] = "*** WARNING *** Aluminum should be limited to 0.05 mg/L to prevent aluminum fouling. Please refer to our Pretreatment Guides for more information.";
            Messages[18] = "*** WARNING *** SDI should be limited to 5.0 to prevent fouling. Please investigate necessary means of pretreatment to lower turbidity levels.";
            Messages[19] = "*** WARNING *** Turbidity should be limited to 1.0 NTU to prevent fouling. Please investigate necessary means of pretreatment to lower turbidity levels.";

            if (Satindex < 1.0 && CaSulfateSat < 150.0 && StronSulfateSat < 200.0 && BariumSulfateSat < 200.0 && CaFluorideSat < 200.0 && SilicaSat < 100.0)
                sTripol = "8500";
            else if (Satindex < 1.8 && CaSulfateSat < 250.0 && StronSulfateSat < 800.0 && BariumSulfateSat < 7000.0 && CaFluorideSat < 8000.0 && SilicaSat < 100.0)
                sTripol = "8500";
            else if (Satindex < 2.0 && CaSulfateSat < 225.0 && StronSulfateSat < 600.0 && BariumSulfateSat < 6000.0 && CaFluorideSat < 10000.0 && SilicaSat < 100.0)
                sTripol = "9000";
            else
                sTripol = "9500";

            if (Satindex <= 0.0)
                Base_Dose = 0.0;
            else
                Base_Dose = 1.0 + 1.5 * Satindex;

            if (CaSulfateSat > 100.0)
                Base_Dose = Base_Dose + 0.5;

            if (StronSulfateSat > 200.0)
                Base_Dose = Base_Dose + 0.5;

            if (BariumSulfateSat > 1000.0)
                Base_Dose = Base_Dose + 0.5;

            if (CaFluorideSat > 100)
                Base_Dose = Base_Dose + 0.5;

            ppmSi = 0.003 * SilicaSat;
            Base_Dose = Base_Dose + ppmSi;
            if (Base_Dose > 8)
                Base_Dose = 8.0;

            FeedFlowforDose = model.Pass[passNumber].CalcData.flow_feed[0] - (model.Pass[0].CalcData.RecircFlowOriginal / 1440.0);
            RecDose = model.Pass[0].Recovery / 100;
            RecircDoseFlow = model.Pass[0].CalcData.RecircFlowOriginal / 1440.0;

            if (model.Pass[0].CalcData.RecircFlowOriginal > 0.0)
                Base_Dose = Base_Dose * (((FeedFlowforDose + RecircDoseFlow) * (1 - RecDose)) / (FeedFlowforDose * (1 - RecDose) + RecircDoseFlow));

            Inj_Rate = Base_Dose * 0.000001 * FeedFlowforDose * 3785.0;

            if (Inj_Rate < 2)
            {
                Solution = Inj_Rate / 3.0;
                Mix = 10.0 * Solution;
                Water = 10.0 - Mix;
            }

            bool isMetric = model.Pass[0].UnitType == "metric" ? true : false;
            if (!isMetric)
                Daily_Usage = (Inj_Rate * 1440.0) / 3785.0;
            else
                Daily_Usage = (Inj_Rate * 1440.0) / 1000.0;

            Month_Usage = Daily_Usage * 30.0;
            var acidType = "None";
            switch (model.Pass[0].ChemType)
            {
                case "H2SO4":
                    acidType = "H2SO4 98%";
                    break;
                case "HCl":
                    acidType = "HCl 37%";
                    break;
                case "NaOH":
                    acidType = "NaOH 30%";
                    break;
            }
            S = $"For optimum system performance, treatment should include Tripol {sTripol} series antiscalant at a dosage rate of {Base_Dose.ToString("0.0")} ppm. ";

            if (!isMetric)
            {
                if (Inj_Rate >= 2.0)
                {
                    S += $"Inject {Inj_Rate.ToString("0.0")} ml/min into the RO feedstream. ";
                }
                else
                {
                    S += " You will be required to mix a diluted solution of antiscalant. ";
                    S += $"For every 10 gallons of volume in your solution tank, mix {Mix.ToString("0.0")} gallons of ";
                    S += $"antiscalant with {Water.ToString("0.0")} gallons of RO permeate water. ";
                    S += "Inject this solution into the RO feedstream at a rate of 3 ml/min. ";
                }

                S += "Antiscalant should be injected at a point where there is a turbulent ";
                S += "flow prior to or just after the pre-RO cartridge filters. The most common ";
                S += "injection point is just prior to or just after the pre-RO cartridge filters.";
                S += Environment.NewLine + Environment.NewLine;
                S += "Operating 24 hours per day, the maximum daily antiscalant usage ";
                S += $"will be {Daily_Usage.ToString("0.0")} gallons. Operating 24 hours per day for 30 days each month, ";
                S += $"the maximum monthly antiscalant usage will be {Month_Usage.ToString("0.0")} gallons.";
                S += Environment.NewLine + Environment.NewLine;
                v = $"Operating 24 hours per day, the maximum daily usage of {acidType} will ";
                v += $"be {model.Pass[0].CalcData.dos_daily.ToString("0.0")} pounds.";
                v += Environment.NewLine;
                v += $"Operating 30 days per month the monthly usage of {acidType} will ";
                v += $"be {model.Pass[0].CalcData.dos_monthly.ToString("0.0")} pounds.";
                v += Environment.NewLine + Environment.NewLine;
            }
            else
            {
                if (Inj_Rate >= 2.0)
                {
                    S += $"Inject {Inj_Rate.ToString("0.0")} ml/min into the RO feedstream. ";
                }
                else
                {
                    S += $"Inject {Inj_Rate.ToString("0.0")} ml/min into the RO feedstream. ";
                    S += " You will be required to mix a diluted solution of antiscalant. ";
                    S += $"For every 38 liters of volume in your solution tank, mix {Mix.ToString("0.0")} liters of ";
                    S += $"antiscalant with {Water.ToString("0.0")} liters of RO permeate water. ";
                    S += "Inject this solution into the RO feedstream at a rate of 3 ml/min. ";
                }
                S += "Antiscalant should be injected at a point where there is a turbulent ";
                S += "flow prior to or just after the pre-RO cartridge filters. The most common ";
                S += "injection point is just prior to or just after the pre-RO cartridge filters.";
                S += Environment.NewLine + Environment.NewLine;
                S += "Operating 24 hours per day, the maximum daily antiscalant usage ";
                S += $"will be {Daily_Usage.ToString("0.0")} liters. Operating 24 hours per day for 30 days each month, ";
                S += $"the maximum monthly antiscalant usage will be {Month_Usage.ToString("0.0")} liters.";
                S += Environment.NewLine + Environment.NewLine;
                v = $"Operating 24 hours per day, the maximum daily usage of {acidType} will ";
                v += $"be {model.Pass[0].CalcData.dos_daily.ToString("0.0")} kgs.";
                v += Environment.NewLine;
                v += $"Operating 30 days per month the monthly usage of {acidType} will ";
                v += $"be {model.Pass[0].CalcData.dos_monthly.ToString("0.0")} kgs.";
                v += Environment.NewLine + Environment.NewLine;
            }

            if (Satindex <= 2.2 && Satindex > 0.0)
            {
                Statuses[0] = true;
            }
            if (Satindex > 2.2)
            {
                Statuses[1] = true;
                Warning = true;
            }
            if (CaSulfateSat > 100.0 && CaSulfateSat <= 250.0)
            {
                Statuses[2] = true;
            }
            if (CaSulfateSat > 250.0)
            {
                Statuses[3] = true;
                Warning = true;
            }
            if (StronSulfateSat > 100.0 && StronSulfateSat <= 800.0)
            {
                Statuses[4] = true;
            }
            if (StronSulfateSat > 800.0)
            {
                Statuses[5] = true;
                Warning = true;
            }
            if (BariumSulfateSat > 100.0 && BariumSulfateSat <= 7000.0)
            {
                Statuses[6] = true;
            }
            if (BariumSulfateSat > 7000.0)
            {
                Statuses[7] = true;
                Warning = true;
            }
            if (CaFluorideSat > 100.0 && CaFluorideSat <= 10000.0)
            {
                Statuses[8] = true;
            }
            if (CaFluorideSat > 10000.0)
            {
                Statuses[9] = true;
                Warning = true;
            }
            if (SilicaSat > 125.0)
            {
                Statuses[10] = true;
            }
            if (SilicaSat > 100.0 && SilicaSat < 125.0)
            {
                Statuses[11] = true;
                Warning = true;
            }
            if (model.Pass[passNumber].CalcData.BetaWarning == 1)
            {
                Statuses[12] = true;
            }
            if (!isMetric)
            {
                if (model.Pass[passNumber].CalcData.PnetiWarning > 0.0)
                {
                    Statuses[13] = true;
                }
                else
                if (model.Pass[passNumber].CalcData.PnetiWarning > 0.0)
                {
                    Statuses[14] = true;
                }
            }
            if (model.Iron > 0.05)
                Statuses[15] = true;
            if(model.Manganese > 0.05)
                Statuses[16] = true;
            if (model.Aluminum > 0.05)
                Statuses[17] = true;
            if(model.SaltDensityIndex > 5.0)
                Statuses[18] = true;
            if (model.Turbidity > 1.0)
                Statuses[19] = true;

            u += Environment.NewLine;

            for (int k = 0; k <= 19; k++)
            {
                if (Statuses[k])
                    u += Messages[k] + Environment.NewLine;
            }

            if (Warning)
                u += Environment.NewLine + "Maximum saturations may sometimes be exceeded. Please contact MICRODYN-NADIR for more information.";

            if (sScalant != "")
            {
                sScalant = sTripol;
                d1 = Base_Dose;
                d2 = Mix;
                d3 = Water;
                d4 = Daily_Usage;
                d5 = Month_Usage;
            }
        }

        private void DebugOut(ProjectionModel model, int passNumber)
        {
            Debug.WriteLine("DebugOut()");
            Debug.WriteLine($"pressf:   { model.Pass[passNumber].CalcData.pressf}");
            Debug.WriteLine($"deltap:   { model.Pass[passNumber].CalcData.deltap}");
            Debug.WriteLine($"flow_bav: { model.Pass[passNumber].CalcData.flow_bav}");
            Debug.WriteLine($"flow_ft:  { model.Pass[passNumber].CalcData.flow_ft}");
            Debug.WriteLine($"flow_pt:  { model.Pass[passNumber].CalcData.flow_pt}");
            Debug.WriteLine($"pneti:    { model.Pass[passNumber].CalcData.pneti}");
            Debug.WriteLine($"press_l:  { model.Pass[passNumber].CalcData.press_l}");
            Debug.WriteLine($"press_h:  { model.Pass[passNumber].CalcData.press_h}");
            Debug.WriteLine($"");
        }

    }
}
