﻿using System;
using System.Collections.Generic;
using System.Reflection;
//using DinkToPdf;
//using DinkToPdf.Contracts;
using ElectronCgi.DotNet;

using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.DependencyInjection;
using Microsoft.Extensions.Logging;

using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using NLog;
using NLog.Extensions.Logging;
using ServiceApp.Enums;
using ServiceApp.Extensions;
using ServiceApp.Models;
using ServiceApp.Services;

namespace ServiceApp
{
    class Program
    {
        private static Connection _connection;
        private static NLog.ILogger _logger;
        private static ISettingsService _settingsService;
        private static IProjectionService _projectionService;
        private static IRawWaterService _rawWaterService;
        private static IDesignCalculationService _designCalculationService;
        private static IReportService _reportService;

        static void Main(string[] args)
        {
            AppInit();

            if (args != null && args.Length > 0 && args[0].Trim() == "--version")
            {
                Console.WriteLine($"ROAM Application Version: {Assembly.GetExecutingAssembly().GetName().Version.ToString()}");
                return;
            }

            //if (args != null && args.Length > 0 && args[0] == "test-design-calc")
            //{
            //    var projection = new ProjectionModel();
            //    projection.Temperature = 12;
            //    projection.TemperatureUnit = "C";
            //    projection.pH = 12;
            //    projection.Ions[0].Value = 12;
            //    projection.Ions[0].UnitType = "ppm";
            //    projection = (ProjectionModel)_rawWaterService.AutoBalance(projection).Data;
            //    _projectionService.SaveProjection("C:\\Users\\mike\\AppData\\Local\\Microdyn-Nadir\\ROAM\\Test1", projection);
            //    projection = (ProjectionModel)_projectionService.LoadProjection("C:\\Users\\mike\\AppData\\Local\\Microdyn-Nadir\\ROAM\\Test1").Data;

            //    projection.EnablePass2 = true;
            //    projection.Pass[0].UnitType = "imperial";
            //    projection.Pass[0].Recovery = 50;
            //    projection.Pass[0].FeedTemp = projection.Temperature;
            //    projection.Pass[0].FeedTempUnit = projection.TemperatureUnit;
            //    projection.Pass[0].RawWaterpH = projection.pH;
            //    projection.Pass[0].AdjustedpH = projection.pH;
            //    projection.Pass[0].ProductFlow = 45;
            //    projection.Pass[0].FluxDecline = 0.025;
            //    projection.Pass[0].SaltPassageIncreaseFactor = 1.3;
            //    projection.Pass[0].ChemType = "H2SO4";
            //    projection.Pass[0].RecircFlow = 0;

            //    var stage = new ProjectionModel.StageData();
            //    stage.StageNumber = 1;
            //    stage.NumberPressureVessels = 1;
            //    stage.PermeatePressure = 50;
            //    stage.ElementsVessel = 2;
            //    stage.ElementSize = 4;
            //    stage.Model = "4040-BW";
            //    projection.Pass[0].StageData.Add(stage);

            //    stage = new ProjectionModel.StageData();
            //    stage.StageNumber = 2;
            //    stage.NumberPressureVessels = 3;
            //    stage.PermeatePressure = 20;
            //    stage.ElementsVessel = 4;
            //    stage.ElementSize = 4;
            //    stage.Model = "4040-LE";
            //    projection.Pass[0].StageData.Add(stage);

            //    stage = new ProjectionModel.StageData();
            //    stage.StageNumber = 3;
            //    stage.NumberPressureVessels = 5;
            //    stage.PermeatePressure = 10;
            //    stage.ElementsVessel = 6;
            //    stage.ElementSize = 4;
            //    stage.Model = "4040-LE";
            //    projection.Pass[0].StageData.Add(stage);

            //    //projection.Pass[0].Recovery = 50;
            //    //projection.Pass[0].FeedTemp = projection.Temperature;
            //    //projection.Pass[0].FeedTempUnit = projection.TemperatureUnit;
            //    //projection.Pass[0].RawWaterpH = projection.pH;
            //    //projection.Pass[0].AdjustedpH = projection.pH;
            //    projection.Pass[1].ProductFlow = 35;
            //    projection.Pass[1].FluxDecline = 0.0;
            //    projection.Pass[1].SaltPassageIncreaseFactor = 0.0;
            //    projection.Pass[1].ChemType = "H2SO4";
            //    projection.Pass[1].RecircFlow = 0.0;

            //    stage = new ProjectionModel.StageData();
            //    stage.StageNumber = 1;
            //    stage.NumberPressureVessels = 2;
            //    stage.PermeatePressure = 40;
            //    stage.ElementsVessel = 6;
            //    stage.ElementSize = 4;
            //    stage.Model = "4040-BW";
            //    projection.Pass[1].StageData.Add(stage);

            //    projection = (ProjectionModel)_designCalculationService.Calc(projection).Data;

            //    _projectionService.SaveProjection("C:\\Users\\mike\\AppData\\Local\\Microdyn-Nadir\\ROAM\\Test1", projection);

            //    //_reportService.GenerateReport(projection);

            //    return;
            //}

            // Electron CGI            
            var electronCGILogPath = Environment.ExpandEnvironmentVariables("%userprofile%\\AppData\\Local\\Microdyn-Nadir-ROAM\\logs\\electroncgi.log");
            _logger.Info($"electronCGI log path: {electronCGILogPath}");
            _connection = new ConnectionBuilder()
                .WithLogging(electronCGILogPath)
                .Build();

            InitAppHandlers();
            InitProjectionHandlers();
            InitTestHandlers();
            InitSettingHandlers();
            InitReportHandlers();

            _connection.Listen();
        }

        private static ServiceResult AppInit()
        {
            // Setup dependency injection with services
            var serviceProvider = new ServiceCollection()
              .AddLogging()
              .AddSingleton<ISettingsService, SettingsService>()
              .AddTransient<IComputationService, ComputationService>()
              .AddTransient<IProjectionService, ProjectionService>()
              .AddTransient<IRawWaterService, RawWaterService>()
              .AddTransient<IDesignCalculationService, DesignCalculationService>()
              .AddTransient<IReportService, ReportService>()
              .AddTransient<ILogService, LogService>()
              //.AddSingleton(typeof(IConverter), new SynchronizedConverter(new PdfTools()))
              .BuildServiceProvider();

            var logService = new LogService();
            logService.InitLogger(NLog.LogLevel.Info);
            var factory = serviceProvider.GetService<ILoggerFactory>();
            factory.AddNLog();

            _logger = LogManager.GetCurrentClassLogger();

            _logger.Info("ElectronCGI AppInit(): Entered.");

            _projectionService = serviceProvider.GetService<IProjectionService>();
            _settingsService = serviceProvider.GetService<ISettingsService>();
            _rawWaterService = serviceProvider.GetService<IRawWaterService>();
            _designCalculationService = serviceProvider.GetService<IDesignCalculationService>();
            _reportService = serviceProvider.GetService<IReportService>();

            Exception settingsException = null;
            var serviceResult = new ServiceResult();
            try
            {
                serviceResult = _settingsService.LoadSettings();
            }
            catch (Exception ex)
            {
                settingsException = ex;
            }

            // If settings threw an exception then we can now log it and exit.
            if (settingsException != null || serviceResult.Result != Result.Success)
            {
                _logger.Fatal(settingsException, "Main(): Exception loading settings.");
                return new ServiceResult(Result.Critical);
            }

            logService.UpdateLogLevel(NLog.LogLevel.FromString(_settingsService.Settings.LogLevel));

            _logger.Info($"ElectronCGI AppInit(): Settings service initialized successfully.");

            logService.DeleteExpiredLogs();

            return new ServiceResult(Result.Success);
        }

        private static void InitAppHandlers()
        {
            _connection.On<string, ServiceResult>("AppInit", data =>
            {
                return AppInit();
            });
        }

        private static void InitProjectionHandlers()
        {
            _connection.On<string, ServiceResult>("LoadProjection", data =>
            {
                _logger.Info("Main(): ElectronCGI LoadProjection called.");
                var result = _projectionService.LoadProjection(data);
                _logger.Info("Main(): ElectronCGI LoadProjection completed.");
                return result;
            });

            _connection.On<string, ServiceResult>("SaveProjection", data =>
            {
                _logger.Info("Main(): ElectronCGI SaveProjection called.");

                JObject jsonData = JObject.Parse(data);
                var path = jsonData["filePath"].ToString();
                var projection = jsonData["data"].ToObject<ProjectionModel>();
                projection.SetIonTypes();

                return _projectionService.SaveProjection(path, projection);
            });

            _connection.On<string, ServiceResult>("CalculateRawWater", data =>
            {
                _logger.Info("Main(): ElectronCGI CalculateRawWater called.");

                var projection = JsonConvert.DeserializeObject<ProjectionModel>(data);
                projection.SetIonTypes();

                return _rawWaterService.Calculate(projection);
            });

            _connection.On<string, ServiceResult>("AutoBalanceRawWater", data =>
            {
                _logger.Info("Main(): ElectronCGI AutoBalanceRawWater called.");

                var projection = JsonConvert.DeserializeObject<ProjectionModel>(data);
                projection.SetIonTypes();

                return _rawWaterService.AutoBalance(projection);
            });

            _connection.On<string, ServiceResult>("InitDesignCalculation", data =>
            {
                _logger.Info("Main(): ElectronCGI InitDesignCalculation called.");

                var projection = JsonConvert.DeserializeObject<ProjectionModel>(data);
                projection.SetIonTypes();

                return _designCalculationService.Init(projection);
            });

            _connection.On<string, ServiceResult>("CalcDesignCalculation", data =>
            {
                _logger.Info("Main(): ElectronCGI CalcDesignCalculation called.");
                _logger.Info(data);
                var projection = JsonConvert.DeserializeObject<ProjectionModel>(data);
                projection.SetIonTypes();

                var result = _designCalculationService.Calc(projection);
                _logger.Info("Main(): ElectronCGI CalcDesignCalculation completed.");
                //_logger.Info(JsonConvert.SerializeObject(result));
                return result;
            });

            //_connection.On<string, ServiceResult>("GenerateReport", data =>
            //{
            //    _logger.Info("Main(): ElectronCGI GenerateReport called.");

            //    var projection = JsonConvert.DeserializeObject<ProjectionModel>(data);
            //    projection.SetIonTypes();

            //    return _reportService.GenerateReport(projection);
            //});
        }

        private static void InitReportHandlers()
        {
            _connection.On<string, ServiceResult>("SaveReport", data =>
            {
                _logger.Info("Main(): ElectronCGI SaveReport called.");
                var reportData = JsonConvert.DeserializeObject<ReportModel>(data);
                var result = _reportService.SaveReport(reportData);
                _logger.Info("Main(): ElectronCGI SaveReport completed.");
                return result;
            });
        }

        private static void InitSettingHandlers()
        {
            _connection.On<string, string>("GetSettings", data =>
            {
                _logger.Info("Main(): ElectronCGI GetSettings called.");

                return JsonConvert.SerializeObject(_settingsService.Settings);
            });

            _connection.On<string, ServiceResult>("PostSettings", data =>
            {
                _logger.Info("Main(): ElectronCGI PostSettings called.");

                var settings = JsonConvert.DeserializeObject<IDictionary<string, string>>(data.Replace("\\", "\\\\"));

                return _settingsService.SetSetting(settings);
            });
        }

        private static void InitTestHandlers()
        {
            _connection.On<string, string>("Ping", data =>
            {
                _logger.Info("Main(): ElectronCGI Ping called.");

                return $"Hi {data}! It is {DateTime.Now}";
            });
        }
    }
}
