﻿using System;
using System.Collections.Generic;
using System.Reflection;
using System.Text;

using ServiceApp.Enums;

namespace ServiceApp.Models
{
    public class ProjectionModel
    {
        public string AppVersion { get; set; }
        public DateTime CalcDate { get; set; }
        public bool WaterIsBalanced { get; set; }
        public bool IsModified { get; set; }
        public bool bCalc { get; set; }

        public double Temperature { get; set; }
        public string TemperatureUnit { get; set; }
        public double pH { get; set; }
        public double Manganese { get; set; }
        public double Iron { get; set; }
        public double SaltDensityIndex { get; set; }
        public double Turbidity { get; set; }
        public double Conductivity { get; set; }
        public double Aluminum { get; set; }
        public double CarbonDioxide { get; set; }

        public List<IonModel> Ions;
        public List<double> VPPM;
        public List<double> VCacO3;
        public List<double> VMilli;

        // SubSummation
        public double CationPpm { get; set; }
        public double CationCacO3 { get; set; }
        public double CatMeq { get; set; }

        public double AnionPpm { get; set; }
        public double AnionCacO3 { get; set; }
        public double AnMeq { get; set; }

        // Totals
        public double CationsTotal { get; set; }
        public double AnionsTotal { get; set; }

        // Saturation Values
        public double KspCaSO4 { get; set; }
        public double KspSrSO4 { get; set; }
        public double KspBaSO4 { get; set; }
        public double KspCaF2 { get; set; }
        public double LSI { get; set; }
        public double SDSI { get; set; }
        public double TDS { get; set; }
        public double OsmoticPressure { get; set; }
        public double SilicaSat { get; set; }

        // Other
        public double ionStrength { get; set; }

        // Design Calculation
        public bool EnablePass2 { get; set; }
        public double OverallSystemRecovery { get; set; }  // Used in two pass only
        public PassData[] Pass { get; set; }
        public WaterSaturationData RawWater { get; set; }
        public WaterSaturationData FeedWater { get; set; }
        public WaterSaturationData Pass1Conc { get; set; }
        public WaterSaturationData Pass2Conc { get; set; }

        //public Dictionary<string, ElementModel> Membrans { get; set; }

        public ProjectionModel()
        {
            Pass = new PassData[] { new PassData(), new PassData() };           

            RawWater = new WaterSaturationData();
            FeedWater = new WaterSaturationData();
            Pass1Conc = new WaterSaturationData();
            Pass2Conc = new WaterSaturationData();

            KspCaSO4 = 0.0;
            KspSrSO4 = 0.0;
            KspBaSO4 = 0.0;
            KspCaF2 = 0.0;

            WaterIsBalanced = false;
            IsModified = false;

            Ions = new List<IonModel>();
            VPPM = new List<double>();
            VCacO3 = new List<double>();
            VMilli = new List<double>();

            var ionList = new List<string>();
            ionList.Add("Calcium");
            ionList.Add("Magnesium");
            ionList.Add("Sodium");
            ionList.Add("Potassium");
            ionList.Add("Ammonium");
            ionList.Add("Barium");
            ionList.Add("Strontium");

            foreach (string ion in ionList)
            {
                //Ions.Add(new IonModel() { Name = ion, IonType = IonType.Cation });
                VPPM.Add(0.0);
                VCacO3.Add(0.0);
                VMilli.Add(0.0);
            }
            ionList.Clear();

            ionList.Add("Carbonate");
            ionList.Add("Bicarbonate");
            ionList.Add("Sulfate");
            ionList.Add("Chloride");
            ionList.Add("Fluoride");
            ionList.Add("Nitrate");
            ionList.Add("Silica");

            foreach (string ion in ionList)
            {
                //Ions.Add(new IonModel() { Name = ion, IonType = IonType.Anion });
                VPPM.Add(0.0);
                VCacO3.Add(0.0);
                VMilli.Add(0.0);
            }
        }

        public void SetIonTypes()
        {
            foreach (var ion in Ions)
            {
                switch (ion.Name)
                {
                    case "Calcium":
                    case "Magnesium":
                    case "Sodium":
                    case "Potassium":
                    case "Ammonium":
                    case "Barium":
                    case "Strontium":
                        ion.IonType = IonType.Cation;
                        break;

                    case "Carbonate":
                    case "Bicarbonate":
                    case "Sulfate":
                    case "Chloride":
                    case "Fluoride":
                    case "Nitrate":
                    case "Silica":
                        ion.IonType = IonType.Anion;
                        break;
                }
            }
        }

        public class PassData
        {
            public CalculatedData CalcData { get; set; }
            public string UnitType { get; set; }
            public string ProjectionBy { get; set; }
            public double Recovery { get; set; }
            public double FeedTemp { get; set; }
            public string FeedTempUnit { get; set; }
            public double RawWaterpH { get; set; }
            public double? AdjustedpH { get; set; }
            public double ProductFlow { get; set; }
            public string ChemType { get; set; }
            public int Age { get; set; }
            public double FluxDecline { get; set; }
            public double SaltPassageIncreaseFactor { get; set; }
            public double RecircFlow { get; set; }
            public double BlendFlow { get; set; }
            public double SystemFlux { get; set; }
            public double FeedCO2 { get; set; }
            public double ChemDosage { get; set; }
            public string Warnings { get; set; }

            public List<StageData> StageData { get; set; }


            //public int Stage2NumberPressureVessels { get; set; }
            //public double Stage2BoostPressure { get; set; }
            //public double Stage2PermeatePressure { get; set; }
            //public int Stage2ElementsVessel { get; set; }
            //public int Stage2ElementSize { get; set; }
            //public string Stage2Model { get; set; }

            //public int Stage3NumberPressureVessels { get; set; }
            //public double Stage3BoostPressure { get; set; }
            //public double Stage3PermeatePressure { get; set; }
            //public int Stage3ElementsVessel { get; set; }
            //public int Stage3ElementSize { get; set; }
            //public string Stage3Model { get; set; }

            //public int Stage4NumberPressureVessels { get; set; }
            //public double Stage4BoostPressure { get; set; }
            //public double Stage4PermeatePressure { get; set; }
            //public int Stage4ElementsVessel { get; set; }
            //public int Stage4ElementSize { get; set; }
            //public string Stage4Model { get; set; }

            public PassData()
            {
                CalcData = new CalculatedData();
                StageData = new List<StageData>();
            }
        }

        public class WaterSaturationData
        {
            public double LSI { get; set; }
            public double SDSI { get; set; }
            public double KspCaSO4 { get; set; }
            public double KspSrSO4 { get; set; }
            public double KspBaSO4 { get; set; }
            public double KspCaF2 { get; set; }
            public double SilicaSat { get; set; }
            public double Fe { get; set; }
            public double Al { get; set; }
            public double OsmoticPressure { get; set; }            
        }

        public class StageData
        {
            public int StageNumber { get; set; }
            public int NumberPressureVessels { get; set; }
            public double BoostPressure { get; set; }
            public double PermeatePressure { get; set; }
            public int ElementsVessel { get; set; }
            public int ElementSize { get; set; }
            public string Model { get; set; }

            public double AverageFlux { get; set; }
            public double FeedPressure { get; set; }
            public double ConcentratedPressure { get; set; }
            public double FeedFlow { get; set; }
            public double ConcentratedFlow { get; set; }
            public double BetaLastElement { get; set; }
            public double ProductSalinity { get; set; }

            public List<ElementData> ElementData { get; set; }

            public StageData()
            {
                ElementData = new List<ElementData>(); 
            }
        }

        public class ElementData
        {
            public int ElementNumber { get; set; }
            public double PermeateFlow { get; set; }
            public double TDS { get; set; }
            public double Beta { get; set; }
            public double Flux { get; set; }
            public double NDPressure { get; set; }
        }

        public class CalculatedData
        {
            public int[] Tube { get; set; }
            public int[] El_tu { get; set; }
            public double[] Ppress { get; set; }
            public double[] BoostPress { get; set; }
            public double[] pExp { get; set; }
            public double[,] pnetiElem { get; set; }
            public double[,] betaElem { get; set; }
            public double[,] flowElem { get; set; }
            public double[,] fluxElem { get; set; }
            public double[,] tdsElem { get; set; }

            public List<IonDataTypeModel> Ion_Data { get; set; }
            public List<IonDataTypeModel> Ion_Data_Perm { get; set; }
            
            public double[] t_ion_ppm { get; set; }
            public double[] ion_ppm { get; set; }
            public double[] ion_ppmc { get; set; }
            public double[] ion_ppmp { get; set; }
            public double[] ion_ppmb { get; set; }
            public double[] ion_ppmf { get; set; }
            public double[] pnetn { get; set; }
            public string[] PANumd { get; set; }
            public double[] col { get; set; }
            public double[] Rej_Ions { get; set; }
            public double[] area { get; set; }
            public double[] leak { get; set; }
            public double[] pcoef { get; set; }
            public double[] atc { get; set; }
            public double[] afact { get; set; }
            public double[] prod_sn { get; set; }
            public double[] prod_snt { get; set; }
            public double[] beta { get; set; }
            public double[] scp { get; set; }
            public double[] press_feed { get; set; }
            public double[] flow_feed { get; set; }
            public double[] scpk { get; set; }
            public double[] b_salp { get; set; }
            public double[] b_prod { get; set; }
            public double[] rej { get; set; }

            public double Recirc { get; set; }
            public double fc022 { get; set; }
            public double Feed_Ret { get; set; }
            public double Con_Ret { get; set; }
            public double cph { get; set; }
            public double pph { get; set; }
            public double RecircFlow { get; set; }
            public double dos { get; set; }
            public double RetStatus { get; set; }
            public double cph2 { get; set; }
            public double pph2 { get; set; }
            public double age { get; set; }
            public double bfact { get; set; }
            public double ftemp { get; set; }
            public double TCor { get; set; }
            public double PProd { get; set; }
            public double sp_fact { get; set; }
            public double fee { get; set; }
            public double con { get; set; }
            public double fco3 { get; set; }
            public double fco2 { get; set; }
            public double fhco3 { get; set; }
            public double acids { get; set; }
            public double acidc { get; set; }
            public double caustic { get; set; }
            public double fcl { get; set; }
            public double fso4 { get; set; }
            public double fna { get; set; }
            public double prod_toti { get; set; }
            public double prod_totp { get; set; }
            public double deltapav { get; set; }
            public double osm_c { get; set; }
            public double feedav { get; set; }
            public double pnetn_av { get; set; }
            public double flow_ft { get; set; }
            public double rec_l { get; set; }
            public double rec_h { get; set; }
            public double rec_av { get; set; }
            public double pressf { get; set; }
            public double Rec { get; set; }
            public double p_salp { get; set; }
            public double p_prod { get; set; }
            public double flow_f { get; set; }
            public double flow_pt { get; set; }
            public double press_l { get; set; }
            public double press_h { get; set; }
            public double flow_b { get; set; }
            public double flow_bav { get; set; }
            public double deltap { get; set; }
            public double pneti { get; set; }
            public double prod { get; set; }
            public double feed { get; set; }
            public double sumfp { get; set; }
            public double fdecl { get; set; }
            public double avsp { get; set; }
            public double osma { get; set; }
            public double osmb { get; set; }
            public double tdsf { get; set; }
            public double avg_ppress { get; set; }
            public double ppress_sum { get; set; }
            public double ftemp2 { get; set; }
            public double pprod2 { get; set; }
            public double Blend { get; set; }
            public double recint { get; set; }
            public double start { get; set; }
            public double elapsed { get; set; }
            public double ph { get; set; }
            public double fph { get; set; }
            public double feedflow { get; set; }
            public int nbanks { get; set; }
            public double el_tot { get; set; }
            public int[] el_bank { get; set; }
            public string str { get; set; }
            public string ptr { get; set; }
            public string ptr2 { get; set; }
            public double tdsc { get; set; }
            public double TotalArea { get; set; }
            public int PnetiWarning { get; set; }
            public int BetaWarning { get; set; }
            public double dos_daily { get; set; }
            public double dos_monthly { get; set; }
            public double SystemFlux { get; set; }
            public double RecircFlowOriginal { get; set; }

            public CalculatedData()
            {
                Tube = new int[4];
                El_tu = new int[4];
                Ppress = new double[4];
                BoostPress = new double[4];
                pExp = new double[4];
                pnetiElem = new double[7, 8];
                betaElem = new double[7, 8];
                flowElem = new double[7, 8];
                fluxElem = new double[7, 8];
                tdsElem = new double[7, 8];
                
                Ion_Data = GetIonData();
                Ion_Data_Perm = GetIonData();

                t_ion_ppm = new double[14];
                ion_ppmc = new double[14];
                ion_ppmp = new double[14];
                scp = new double[14];
                ion_ppmb = new double[14];
                ion_ppmf = new double[14];
                ion_ppm = new double[14];
                Rej_Ions = new double[14];
                el_bank = new int[7];
                area = new double[7];
                leak = new double[7];
                pcoef = new double[7];
                rej = new double[7];
                atc = new double[7];
                afact = new double[7];
                prod_sn = new double[7];
                prod_snt = new double[7];
                beta = new double[7];
                PANumd = new string[19];
                pnetn = new double[7];
                col = new double[7];
                scp = new double[15];
                press_feed = new double[6];
                flow_feed = new double[5];
                scpk = new double[15];
                b_salp = new double[7];
                b_prod = new double[7];
            }

            private List<IonDataTypeModel> GetIonData()
            {
                // Values from iondata.dat file
                var ions = new List<IonDataTypeModel>();
                ions.Add(new IonDataTypeModel(0.2, 0.2, 2, 40.1, 0.2, 0.087, 0.1, 0.0));
                ions.Add(new IonDataTypeModel(0.2, 0.2, 2, 24.3, 0.2, 0.085, 0.1, 0.0));
                ions.Add(new IonDataTypeModel(1.0, 1.0, 1, 23.0, 1.0, 0.87, 1.0, 0.0));
                ions.Add(new IonDataTypeModel(1.3, 1.3, 1, 39.0, 1.3, 1.07, 1.3, 0.0));
                ions.Add(new IonDataTypeModel(3.0, 1.5, 1, 18.0, 1.5, 1.4, 1.5, 0.0));
                ions.Add(new IonDataTypeModel(0.1, 0.1, 2, 137.3, 0.1, 0.1, 0.02, 0.0));
                ions.Add(new IonDataTypeModel(0.1, 0.1, 2, 87.6, 0.1, 0.08, 0.02, 0.0));
                ions.Add(new IonDataTypeModel(0.1, 0.1, -2, 60.0, 0.1, 0.1, 0.1, 0.0));
                ions.Add(new IonDataTypeModel(1.0, 1.0, -1, 61.0, 1.0, 1.15, 1.0, 0.0));
                ions.Add(new IonDataTypeModel(0.1, 0.1, -2, 96.0, 0.1, 0.025, 0.02, 0.0));
                ions.Add(new IonDataTypeModel(1.0, 1.0, -1, 35.45, 1.0, 1.17, 1.0, 0.0));
                ions.Add(new IonDataTypeModel(0.5, 0.5, -1, 19.0, 0.5, 5, 0.6, 0.0));
                ions.Add(new IonDataTypeModel(2.0, 2.0, -1, 62.0, 2.0, 2.39, 0.1, 0.0));
                ions.Add(new IonDataTypeModel(2.0, 0.7, -1, 60.1, 1.0, 1.1, 1.5, 0.0));
                return ions;
            }
        }

        private Dictionary<string, ElementModel> GetElements()
        {
            var elements = new Dictionary<string, ElementModel>();

            var element = new ElementModel();
            element.Model = "BW";
            element.Diameter = 4;
            element.Length = 40;
            element.Reject = 99.5;
            element.Flow = 2400;
            element.pcoef = 0.12;
            element.pExp = 1.35;
            element.area = 85;
            element.leak = 0.001;
            element.Atc1 = 9.902317;
            element.PSTD = 198;
            element.TCor = 2700;
            elements.Add("4040-BW", element);

            element = new ElementModel();
            element.Model = "LE";
            element.Diameter = 4;
            element.Length = 40;
            element.Reject = 99.3;
            element.Flow = 3350;
            element.pcoef = 0.12;
            element.pExp = 1.35;
            element.area = 85;
            element.leak = 0.001;
            element.Atc1 = 13.82198;
            element.PSTD = 198;
            element.TCor = 2700;
            elements.Add("4040-LE", element);

            element = new ElementModel();
            element.Model = "BW-3";
            element.Diameter = 7.9;
            element.Length = 40;
            element.Reject = 99.5;
            element.Flow = 9700;
            element.pcoef = 0.0124;
            element.pExp = 1.56;
            element.area = 365;
            element.leak = 0.001;
            element.Atc1 = 9.320161;
            element.PSTD = 198;
            element.TCor = 2700;
            elements.Add("8040-BW-365", element);

            element = new ElementModel();
            element.Model = "BW-4";
            element.Diameter = 7.9;
            element.Length = 40;
            element.Reject = 99.5;
            element.Flow = 10700;
            element.pcoef = 0.0124;
            element.pExp = 1.56;
            element.area = 400;
            element.leak = 0.001;
            element.Atc1 = 9.381414;
            element.PSTD = 198;
            element.TCor = 2700;
            elements.Add("8040-BW-400", element);

            element = new ElementModel();
            element.Model = "BW-4";
            element.Diameter = 7.9;
            element.Length = 40;
            element.Reject = 99.5;
            element.Flow = 11700;
            element.pcoef = 0.0124;
            element.pExp = 1.56;
            element.area = 440;
            element.leak = 0.001;
            element.Atc1 = 9.32562;
            element.PSTD = 198;
            element.TCor = 2700;
            elements.Add("8040-BW-440", element);

            element = new ElementModel();
            element.Model = "LE-4";
            element.Diameter = 7.9;
            element.Length = 40;
            element.Reject = 99.3;
            element.Flow = 16000;
            element.pcoef = 0.0124;
            element.pExp = 1.56;
            element.area = 400;
            element.leak = 0.001;
            element.Atc1 = 14.02828;
            element.PSTD = 198;
            element.TCor = 2700;
            elements.Add("8040-LE-400", element);

            element = new ElementModel();
            element.Model = "LE-4";
            element.Diameter = 7.9;
            element.Length = 40;
            element.Reject = 99.3;
            element.Flow = 17539;
            element.pcoef = 0.0124;
            element.pExp = 1.56;
            element.area = 440;
            element.leak = 0.001;
            element.Atc1 = 13.97966;
            element.PSTD = 198;
            element.TCor = 2700;
            elements.Add("8040-LE-440", element);

            return elements;
        }
    }
}
