var PdfPrinter = require('pdfmake');
var fs = require('fs');
var ellipsis = require('text-ellipsis');

module.exports = {
  createReport: async function (fileName, projection, debug = false) {
    return new Promise(function (resolve) {
      console.log("CreateReport()");
      console.log(JSON.stringify(projection));
      // Vars for presentation data
      var calcDate = projection.calcDate.replace(/T/, ' ').replace(/\..+/, '');
      var flowUnit = projection.pass[0].unitType == 'metric' ? 'm3/h' : 'gpm';
      var fluxUnit = projection.pass[0].unitType == 'metric' ? 'lmh' : 'gfd';
      var pressureUnit = projection.pass[0].unitType == 'metric' ? 'bar' : 'psi';
      var unitAdj = projection.pass[0].unitType == 'metric' ? 14.5038 : 1.0;
      var pass1ChemType = "";
      switch (projection.pass[0].chemType) {
        case "H2SO4":
          pass1ChemType = ", H2SO4 98%";
          break;
        case "HCl":
          pass1ChemType = ", HCl 37%";
          break;
        case "NaOH":
          pass1ChemType = ", NaOH 30%";
          break;
      }
      var pass2ChemType = "";
      switch (projection.pass[0].chemType) {
        case "H2SO4":
          pass2ChemType = ", H2SO4 98%";
          break;
        case "HCl":
          pass2ChemType = ", HCl 37%";
          break;
        case "NaOH":
          pass2ChemType = ", NaOH 30%";
          break;
      }

      // PDF Data
      var fonts = {
        // CrimsonRoman: {
        //   normal: __dirname + '/assets/fonts/Crimson-Roman.ttf',
        //   bold: __dirname + '/assets/fonts/Crimson-Bold.ttf',
        //   italics: __dirname + '/assets/fonts/Crimson-Italic.ttf',
        //   bolditalics: __dirname + '/assets/fonts/Crimson-BoldItalic.ttf'
        // },
        'Open Sans': {
          normal: __dirname + '/assets/fonts/OpenSans-Regular.ttf',
          bold: __dirname + '/assets/fonts/OpenSans-Bold.ttf',
          italics: __dirname + '/assets/fonts/OpenSans-RegularItalic.ttf',
          bolditalics: __dirname + '/assets/fonts/OpenSans-BoldItalic.ttf'
        }
      };
      var printer = new PdfPrinter(fonts);

      var borderLight = '#F3F4F5';
      var borderDark = '#D3D3D3';
      var thFill = '#E1E2E4';
      var thSubFill = '#ECECEE';
      var thVertFill = thSubFill;
      var thVertAltFill = '#F5F5F6';
      var thVertAltLightFill = '#FFFFFF';
      var tdFill = '#FFFFFF';
      var tdAltFill = '#F7F7F8';

      var paddingLeft = 0;
      var paddingRight = 0;
      var paddingTop = 0;
      var paddingBottom = 0;
      var cellHMargin = -1.0;
      var cellVMargin = 1.0;

      var docDefinition = docSetup();
      docDefinition.content.push(docSystemDesign(0, projection));
      docDefinition.content.push(docWaterQuality(0, projection));
      docDefinition.content.push(docStageInformation(0, projection));
      docDefinition.content.push(docIndividualElements(0, projection));

      if (projection.enablePass2) {
        docDefinition.content.push(docSystemDesign(1, projection));
        docDefinition.content.push(docWaterQuality(1, projection));
        docDefinition.content.push(docStageInformation(1, projection));
        docDefinition.content.push(docIndividualElements(1, projection));
      }

      docDefinition.content.push(docWaterSaturation(projection));
      docDefinition.content.push(docSolubilityDesignWarning(projection));

      docDefinition.footer = docLastPageFooter();

      // console.log("***** DOC DEFINITION *****");
      // console.log(JSON.stringify(docDefinition.content));
      // console.log("***** DOC DEFINITION *****");

      function docSetup() {
        return {
          pageSize: 'A4',
          pageMargins: [35, 35, 35, 30],
          defaultStyle: { fontSize: 9, font: 'Open Sans', margin: [0, 0, 0, 0], padding: [0, 0, 0.0] },
          styles: {
            logoSvg: {
              margin: [0, 5, 0, 10],
              alignment: 'center'
            },
            section: {
              fontSize: 12,
              bold: true,
              alignment: 'center',
              margin: [0, 11, 0, 0]
            },
            table: {
              margin: [0, 6, 0, 0]
            },
            tableNote: {
              alignment: 'right',
              margin: [0, 2, 0, 0]
            },
            th: {
              bold: true,
              fillColor: thFill,
              alignment: 'center',
              margin: [cellHMargin, cellVMargin, cellHMargin, cellVMargin]
            },
            thSub: {
              fillColor: thSubFill,
              alignment: 'right',
              margin: [cellHMargin, cellVMargin, cellHMargin, cellVMargin]
            },
            thVert: {
              bold: true,
              fillColor: thSubFill,
              margin: [cellHMargin, cellVMargin, cellHMargin, cellVMargin]
            },
            thVertAlt: {
              bold: true,
              fillColor: thVertAltFill,
              margin: [cellHMargin, cellVMargin, cellHMargin, cellVMargin]
            },
            thVertAltLight: {
              bold: true,
              fillColor: thVertAltLightFill,
              margin: [cellHMargin, cellVMargin, cellHMargin, cellVMargin]
            },
            td: {
              fillColor: tdFill,
              alignment: 'right',
              margin: [cellHMargin, cellVMargin, cellHMargin, cellVMargin]
            },
            tdAlt: {
              fillColor: tdAltFill,
              alignment: 'right',
              margin: [cellHMargin, cellVMargin, cellHMargin, cellVMargin]
            },
            tdAltDark: {
              fillColor: thSubFill,
              alignment: 'right',
              margin: [cellHMargin, cellVMargin, cellHMargin, cellVMargin]
            },
            tdAltDarker: {
              fillColor: thFill,
              alignment: 'right',
              margin: [cellHMargin, cellVMargin, cellHMargin, cellVMargin]
            },
            thSmall: {
              fontSize: 8,
              bold: true,
              fillColor: thFill,
              alignment: 'center',
              margin: [0, 0, 0, 0]
            },
            thSubSmall: {
              fontSize: 8,
              fillColor: thSubFill,
              alignment: 'center',
              margin: [0, cellVMargin, 0, cellVMargin]
            },
            tdSmall: {
              fontSize: 8,
              fillColor: tdFill,
              alignment: 'right',
              margin: [0, 0, 0, 0]
            },
            tdAltSmall: {
              fontSize: 8,
              fillColor: tdAltFill,
              alignment: 'right',
              margin: [0, 0, 0, 0]
            },
          },
          content: [
            { svg: logoSvg, style: 'logoSvg', width: 280 },

            { text: 'REVERSE OSMOSIS APPLICATION FOR MICRODYN-NADIR (ROAM)', style: 'section' },
            {
              style: 'table',
              table: {
                widths: [205, 205, '*'],
                body: [
                  [{ text: 'Project Name', style: 'th', alignment: 'left', borderColor: [thFill, thFill, borderLight, borderLight] },
                  { text: 'Projection By', style: 'th', alignment: 'left', borderColor: [thFill, thFill, borderLight, borderLight] },
                  { text: 'Date/time', style: 'th', alignment: 'left', borderColor: [thFill, thFill, thFill, borderLight] }],
                  [{ text: ellipsis(fileName, 40), style: 'td', alignment: 'left' },
                  { text: ellipsis(projection.pass[0].projectionBy, 40), style: 'td', alignment: 'left' },
                  { text: calcDate, style: 'td', alignment: 'left' }],
                ]
              },
              layout: {
                fillColor: function (rowIndex, node, columnIndex) { return (rowIndex % 2 === 0) ? gray : null; },
                hLineWidth: function (i, node) { return 1; },
                vLineWidth: function (i, node) { return 1; },
                hLineColor: function (i, node) { return borderDark; },
                vLineColor: function (i, node) { return borderDark; },
              }
            },
          ]
        };
      }

      function docSystemDesign(passNumber, projection) {
        var pass = projection.pass[passNumber];
        var pass1Label = projection.enablePass2 ? ' - 1st Pass' : '';
        var sectionLabel = passNumber == 0 ? pass1Label : ' - 2nd Pass';
        var pageBreak = passNumber == 0 ? 'none' : 'before';
        var concPressure = pass.stageData[pass.stageData.length - 1].concentratedPressure;
        var section = [{ text: 'System Design' + sectionLabel, style: 'section', pageBreak: pageBreak },
        {
          style: 'table',
          table: {
            widths: [135, '*', 33, 135, '*', 33],
            body: [
            ]
          },
          layout: {
            hLineWidth: function (i, node) { return 1; },
            vLineWidth: function (i, node) { return 1; },
            hLineColor: function (i, node) { return borderDark; },
            vLineColor: function (i, node) { return borderDark; },
          }
        }];

        if (projection.enablePass2 && passNumber == 0)
          pass.recovery = pass.productFlow * 100.0 / (pass.stageData[0].feedFlow - pass.recircFlow);

        // Itemize each label, value, and unit allowing ordering to be different for pass 1 and 2
        var items = new Array(17);
        items[0] = [{ text: 'Permeate Flow:', style: 'thVert' }, { text: `${pass.productFlow.toFixed(1)}`, style: 'tdAlt' }, { text: flowUnit, style: 'tdAlt', alignment: 'left' }];
        items[1] = [{ text: 'System Flux:', style: 'thVert' }, { text: `${pass.systemFlux.toFixed(1)}`, style: 'tdAlt' }, { text: fluxUnit, style: 'tdAlt', alignment: 'left' }];
        items[2] = [{ text: projection.enablePass2 ? 'Pass Recovery:' : 'System Recovery:', style: 'thVertAlt' }, { text: `${pass.recovery.toFixed(1)}`, style: 'td' }, { text: '%', style: 'td', alignment: 'left' }];
        items[3] = [{ text: 'System Recovery:', style: 'thVertAlt' }, { text: `${projection.overallSystemRecovery.toFixed(1)}`, style: 'td' }, { text: '%', style: 'td', alignment: 'left' }];
        items[4] = [{ text: `Chemical Dosage${pass1ChemType}:`, style: 'thVertAlt' }, { text: `${pass.chemDosage.toFixed(1)}`, style: 'td' }, { text: 'ppm', style: 'td', alignment: 'left' }];
        items[5] = [{ text: 'Feedwater Temp:', style: 'thVert' }, { text: `${pass.feedTemp.toFixed(1)}`, style: 'tdAlt' }, { text: `°${pass.feedTempUnit}`, style: 'tdAlt', alignment: 'left' }];
        items[6] = [{ text: 'Feed CO2:', style: 'thVert' }, { text: `${pass.feedCO2.toFixed(1)}`, style: 'tdAlt' }, { text: 'ppm', style: 'tdAlt', alignment: 'left' }];
        items[7] = [{ text: 'Element Age:', style: 'thVertAlt' }, { text: `${pass.age.toFixed(1)}`, style: 'td' }, { text: 'years', style: 'td', alignment: 'left' }];
        items[8] = [{ text: 'Flux Decline Coef.:', style: 'thVertAlt' }, { text: `${pass.fluxDecline}`, style: 'td' }, { text: '%', style: 'td', alignment: 'left' }];
        items[9] = [{ text: 'Raw Water pH:', style: 'thVert' }, { text: `${pass.rawWaterpH.toFixed(1)}`, style: 'tdAlt' }, { text: '', style: 'tdAlt', alignment: 'left' }];
        items[10] = [{ text: '3-yr S.P.I.F:', style: 'thVert' }, { text: `${pass.saltPassageIncreaseFactor.toFixed(1)}`, style: 'tdAlt' }, { text: '', style: 'tdAlt', alignment: 'left' }];
        items[11] = [{ text: 'Feed Pressure:', style: 'thVertAlt' }, { text: `${pass.stageData[0].feedPressure.toFixed(1)}`, style: 'td' }, { text: pressureUnit, style: 'td', alignment: 'left' }];
        items[12] = [{ text: 'Concentrate Pressure:', style: 'thVertAlt' }, { text: `${concPressure.toFixed(1)}`, style: 'td' }, { text: pressureUnit, style: 'td', alignment: 'left' }];
        items[13] = [{ text: 'Recirc. Flow:', style: 'thVert' }, { text: `${pass.recircFlow.toFixed(1)}`, style: 'tdAlt' }, { text: flowUnit, style: 'tdAlt', alignment: 'left' }];
        items[14] = [{ text: 'Blend Flow:', style: 'thVert' }, { text: `${pass.blendFlow.toFixed(1)}`, style: 'tdAlt' }, { text: flowUnit, style: 'tdAlt', alignment: 'left' }];
        items[15] = [{ text: '', style: 'thVert' }, { text: '', style: 'tdAlt' }, { text: '', style: 'tdAlt', alignment: 'left' }];
        items[16] = [{ text: 'Blended Perm. Flow', style: 'thVert' }, { text: `${(pass.productFlow + pass.blendFlow).toFixed(1)}`, style: 'tdAlt' }, { text: flowUnit, style: 'tdAlt', alignment: 'left' }];

        section[1].table.body.push([items[0][0], items[0][1], items[0][2], items[1][0], items[1][1], items[1][2]]);
        if (passNumber == 0 && projection.enablePass2) {
          section[1].table.body.push([items[2][0], items[2][1], items[2][2], items[4][0], items[4][1], items[4][2]]);
          section[1].table.body.push([items[3][0], items[3][1], items[3][2], items[6][0], items[6][1], items[6][2]]);
          section[1].table.body.push([items[5][0], items[5][1], items[5][2], items[8][0], items[8][1], items[8][2]]);
          section[1].table.body.push([items[7][0], items[7][1], items[7][2], items[10][0], items[10][1], items[10][2]]);
          section[1].table.body.push([items[9][0], items[9][1], items[9][2], items[12][0], items[12][1], items[12][2]]);
          section[1].table.body.push([items[11][0], items[11][1], items[11][2], items[14][0], items[14][1], items[14][2]]);
          section[1].table.body.push([items[13][0], items[13][1], items[13][2], items[15][0], items[15][1], items[15][2]]);
        }
        else {
          section[1].table.body.push([items[2][0], items[2][1], items[2][2], items[4][0], items[4][1], items[4][2]]);
          section[1].table.body.push([items[5][0], items[5][1], items[5][2], items[6][0], items[6][1], items[6][2]]);
          section[1].table.body.push([items[7][0], items[7][1], items[7][2], items[8][0], items[8][1], items[8][2]]);
          section[1].table.body.push([items[9][0], items[9][1], items[9][2], items[10][0], items[10][1], items[10][2]]);
          section[1].table.body.push([items[11][0], items[11][1], items[11][2], items[12][0], items[12][1], items[12][2]]);
          section[1].table.body.push([items[13][0], items[13][1], items[13][2], items[14][0], items[14][1], items[14][2]]);
        }

        if (pass.blendFlow > 0.0)
          section[1].table.body.push([items[15][0], items[15][1], items[15][2], items[16][0], items[16][1], items[16][2]]);

        return section;
      }

      function docWaterQuality(passNumber, projection) {
        var pass = projection.pass[passNumber];
        var pass1Label = projection.enablePass2 ? ' - 1st Pass' : '';
        var sectionLabel = passNumber == 0 ? pass1Label : ' - 2nd Pass';
        var section = [{ text: 'Water Quality' + sectionLabel, style: 'section' },
        {
          style: 'table',
          table: {
            widths: ['*', 43, 43, 43, 43, 43, 43, 43, 43],
            headerRows: 1,
            body: [
              [{ text: '', style: 'th', borderColor: [thFill, thFill, borderLight, borderLight] },
              { text: 'Raw Water', style: 'th', colSpan: 2, borderColor: [thFill, thFill, borderLight, borderLight] }, {},
              { text: 'Adjusted Feed Water', margin: [0, 2, 0, 2], style: 'th', colSpan: 2, borderColor: [thFill, thFill, borderLight, borderLight] }, {},
              { text: 'Concentrate', style: 'th', colSpan: 2, borderColor: [thFill, thFill, borderLight, borderLight] }, {},
              { text: 'Permeate', style: 'th', colSpan: 2, borderColor: [thFill, thFill, thFill, borderLight] }, {}],
              [{ text: 'Ion', style: 'thVert', bold: true, alignment: 'left' }, { text: 'mg/l', style: 'thSub' }, { text: 'CaCO3', style: 'thSub' }, { text: 'mg/l', style: 'thSub' },
              { text: 'CaCO3', style: 'thSub' }, { text: 'mg/l', style: 'thSub' }, { text: 'CaCO3', style: 'thSub' }, { text: 'mg/l', style: 'thSub' },
              { text: 'CaCO3', style: 'thSub' }]
            ]
          },
          layout: {
            hLineWidth: function (i, node) { return 1; },
            vLineWidth: function (i, node) { return 1; },
            hLineColor: function (i, node) { return borderDark; },
            vLineColor: function (i, node) { return borderDark; },
          }
        }];

        ions.forEach(function (value, index, array) {
          var labelStyle = index % 2 ? 'thVert' : 'thVertAltLight';
          var cellStyle = index % 2 ? 'tdAlt' : 'td';
          var row = [{ text: `${value.name} (${value.abbr})`, style: labelStyle },
          { text: `${pass.calcData.ion_ppm[index].toFixed(2)}`, style: cellStyle },
          { text: `${(pass.calcData.ion_ppm[index] / value.CaCO3).toFixed(2)}`, style: cellStyle },
          { text: `${pass.calcData.t_ion_ppm[index].toFixed(2)}`, style: cellStyle },
          { text: `${(pass.calcData.t_ion_ppm[index] / value.CaCO3).toFixed(2)}`, style: cellStyle },
          { text: `${pass.calcData.ion_ppmc[index].toFixed(2)}`, style: cellStyle },
          { text: `${(pass.calcData.ion_ppmc[index] / value.CaCO3).toFixed(2)}`, style: cellStyle },
          { text: `${pass.calcData.ion_ppmp[index].toFixed(2)}`, style: cellStyle },
          { text: `${(pass.calcData.ion_ppmp[index] / value.CaCO3).toFixed(2)}`, style: cellStyle },];
          section[1].table.body.push(row);
        });

        var tdsRow = [{ text: 'TDS', style: 'th', alignment: 'left' },
        { text: TDSmeqLTotal(pass.calcData.ion_ppm).toFixed(2), style: 'tdAltDarker' },
        { text: TDSCaCO3Total(pass.calcData.ion_ppm).toFixed(2), style: 'tdAltDarker' },
        { text: TDSmeqLTotal(pass.calcData.t_ion_ppm).toFixed(2), style: 'tdAltDarker' },
        { text: TDSCaCO3Total(pass.calcData.t_ion_ppm).toFixed(2), style: 'tdAltDarker' },
        { text: TDSmeqLTotal(pass.calcData.ion_ppmc).toFixed(2), style: 'tdAltDarker' },
        { text: TDSCaCO3Total(pass.calcData.ion_ppmc).toFixed(2), style: 'tdAltDarker' },
        { text: TDSmeqLTotal(pass.calcData.ion_ppmp).toFixed(2), style: 'tdAltDarker' },
        { text: TDSCaCO3Total(pass.calcData.ion_ppmp).toFixed(2), style: 'tdAltDarker' }];
        section[1].table.body.push(tdsRow);

        var pHRow = [{ text: 'pH', style: 'thVert' },
        { text: pass.rawWaterpH.toFixed(2), style: 'tdAltDark' }, { text: '', style: 'tdAltDark' },
        { text: pass.calcData.fph.toFixed(2), style: 'tdAltDark' }, { text: '', style: 'tdAltDark' },
        { text: pass.calcData.cph.toFixed(2), style: 'tdAltDark' }, { text: '', style: 'tdAltDark' },
        { text: pass.calcData.pph.toFixed(2), style: 'tdAltDark' }, { text: '', style: 'tdAltDark' }];
        section[1].table.body.push(pHRow);

        return section;
      }

      function docStageInformation(passNumber, projection) {
        var pass = projection.pass[passNumber];
        var pass1Label = projection.enablePass2 ? ' - 1st Pass' : '';
        var sectionLabel = passNumber == 0 ? pass1Label : ' - 2nd Pass';
        var section = [{ text: 'Stage Information' + sectionLabel, style: 'section', pageBreak: "before" },
        {
          style: 'table',
          table: {
            widths: [22, 24, 37, '*', 25, 25, 42, 42, 42, 56, 21, 25],
            headerRows: 1,
            body: [
              [{ text: 'Stage', style: 'thSmall', borderColor: [thFill, thFill, borderLight, borderLight] },
              { text: 'Array', style: 'thSmall', borderColor: [thFill, thFill, borderLight, borderLight] },
              { text: 'No of Elements', style: 'thSmall', borderColor: [thFill, thFill, borderLight, borderLight] },
              { text: 'Element', style: 'thSmall', borderColor: [thFill, thFill, borderLight, borderLight] },
              { text: 'Feed Flow', style: 'thSmall', borderColor: [thFill, thFill, borderLight, borderLight] },
              { text: 'Conc. Flow', style: 'thSmall', borderColor: [thFill, thFill, borderLight, borderLight] },
              { text: 'Pressure', style: 'thSmall', colSpan: 2, borderColor: [thFill, thFill, borderLight, borderLight] }, {},
              { text: 'Pressure', style: 'thSmall', colSpan: 2, borderColor: [thFill, thFill, borderLight, borderLight] }, {},
              { text: 'Beta', style: 'thSmall', borderColor: [thFill, thFill, borderLight, borderLight] },
              { text: 'Avg. Flux', style: 'thSmall', borderColor: [thFill, thFill, thFill, borderLight] }],
              [{ text: '', style: 'thSubSmall' }, { text: '', style: 'thSubSmall' }, { text: '', style: 'thSubSmall' }, { text: '', style: 'thSubSmall' },
              { text: `(${flowUnit})`, style: 'thSubSmall' }, { text: `(${flowUnit})`, style: 'thSubSmall' },
              { text: `Feed (${pressureUnit})`, style: 'thSubSmall' },
              { text: `Conc. (${pressureUnit})`, style: 'thSubSmall' },
              { text: `Boost (${pressureUnit})`, style: 'thSubSmall' },
              { text: `Permeate (${pressureUnit})`, style: 'thSubSmall' },
              { text: '', style: 'thSubSmall' },
              { text: `(${fluxUnit})`, style: 'thSubSmall' }]

            ]
          },
          layout: {
            hLineWidth: function (i, node) { return 1; },
            vLineWidth: function (i, node) { return 1; },
            hLineColor: function (i, node) { return borderDark; },
            vLineColor: function (i, node) { return borderDark; },
          }
        },];
        pass.stageData.forEach(function (value, index, array) {
          var cellStyle = index % 2 ? 'tdSmall' : 'tdAltSmall';
          var row = [{ text: value.stageNumber, style: cellStyle, alignment: 'center' },
          { text: `${value.numberPressureVessels}x${value.elementsVessel}`, style: cellStyle, alignment: 'center' },
          { text: value.numberPressureVessels * value.elementsVessel, style: cellStyle, alignment: 'center' },
          { text: value.model, style: cellStyle, alignment: 'center' },
          { text: value.feedFlow.toFixed(1), style: cellStyle },
          { text: value.concentratedFlow.toFixed(1), style: cellStyle },
          { text: value.feedPressure.toFixed(1), style: cellStyle },
          { text: value.concentratedPressure.toFixed(1), style: cellStyle, alignment: 'center' },
          { text: index == 0.00 ? "-" : value.boostPressure.toFixed(2), style: cellStyle, alignment: index == 0.00 ? 'center' : 'right' },
          { text: value.permeatePressure.toFixed(2), style: cellStyle },
          { text: value.betaLastElement.toFixed(2), style: cellStyle },
          { text: value.averageFlux.toFixed(1), style: cellStyle }];
          section[1].table.body.push(row);
        });

        return section;
      }

      function docIndividualElements(passNumber, projection) {
        var pass = projection.pass[passNumber];
        var pass1Label = projection.enablePass2 ? ' - 1st Pass' : '';
        var sectionLabel = passNumber == 0 ? pass1Label : ' - 2nd Pass';
        var section = [{ text: 'Individual Elements' + sectionLabel, style: 'section' },
        {
          style: 'table',
          table: {
            widths: [35, 65, 114, 40, 32, 55, '*'],
            headerRows: 1,
            body: [
              [{ text: 'Stage', style: 'thSmall', borderColor: [thFill, thFill, borderLight, borderLight] },
              { text: 'Element No.', style: 'thSmall', borderColor: [thFill, thFill, borderLight, borderLight] },
              { text: `Permeate Flow (${flowUnit})`, style: 'thSmall', borderColor: [thFill, thFill, borderLight, borderLight] },
              { text: 'TDS', style: 'thSmall', borderColor: [thFill, thFill, borderLight, borderLight] },
              { text: 'Beta', style: 'thSmall', borderColor: [thFill, thFill, borderLight, borderLight] },
              { text: `Flux (${fluxUnit})`, style: 'thSmall', borderColor: [thFill, thFill, borderLight, borderLight] },
              { text: `Net Driving Pressure (${pressureUnit})`, style: 'thSmall', borderColor: [thFill, thFill, thFill, borderLight] }]
            ]
          },
          layout: {
            hLineWidth: function (i, node) { return 1; },
            vLineWidth: function (i, node) { return 1; },
            hLineColor: function (i, node) { return borderDark; },
            vLineColor: function (i, node) { return borderDark; },
          }
        }];

        var cellStyle = "";
        var rowToPageBreak = 29;
        var rowCount = 0;
        var pageBreak = "none";
        pass.stageData.forEach(function (value, index, array) {
          var i = 0;

          for (i = 0; i < value.elementData.length; i++) {
            var element = value.elementData[i];
            cellStyle = cellStyle == 'td' ? 'tdAlt' : 'td';
            rowCount++;
            if (rowCount == rowToPageBreak)
              pageBreak = "before";

            //var row = [{ text: i == 0 ? value.stageNumber : '', style: cellStyle, alignment: 'center' },
            var row = [{ text: value.stageNumber, style: cellStyle, alignment: 'center', pageBreak: pageBreak },
            { text: element.elementNumber, style: cellStyle, alignment: 'center', pageBreak: pageBreak },
            { text: element.permeateFlow.toFixed(2), style: cellStyle, pageBreak: pageBreak },
            { text: element.tds.toFixed(2), style: cellStyle, pageBreak: pageBreak },
            { text: element.beta.toFixed(2), style: cellStyle, pageBreak: pageBreak },
            { text: element.flux.toFixed(2), style: cellStyle, pageBreak: pageBreak },
            { text: element.ndPressure.toFixed(1), style: cellStyle, pageBreak: pageBreak }];
            section[1].table.body.push(row);

            if (rowCount == rowToPageBreak)
              pageBreak = "none";
          }
        });

        // [{ text: '1', style: 'td', alignment: 'center' }, { text: '1', style: 'td', alignment: 'center' }, { text: '0.55', style: 'td' },
        // { text: '0.08', style: 'td' }, { text: '1.09', style: 'td' }, { text: '14.75', style: 'td' }, { text: '10.2', style: 'td' }],
        // [{ text: '1', style: 'tdAlt', alignment: 'center' }, { text: '1', style: 'tdAlt', alignment: 'center' }, { text: '0.55', style: 'tdAlt' },
        // { text: '0.08', style: 'tdAlt' }, { text: '1.09', style: 'tdAlt' }, { text: '14.75', style: 'tdAlt' }, { text: '10.2', style: 'tdAlt' }],

        return section;
      }

      function docWaterSaturation(projection) {
        var sectionLabel = projection.enablePass2 ? ' - 1st & 2nd Pass' : '';
        var section = [{ text: 'Water Saturation Indexes' + sectionLabel, style: 'section', pageBreak: 'before' },
        {
          style: 'table',
          table: {
            widths: [150, '*', '*'],
            headerRows: 1,
            body: []
          },
          layout: {
            hLineWidth: function (i, node) { return 1; },
            vLineWidth: function (i, node) { return 1; },
            hLineColor: function (i, node) { return borderDark; },
            vLineColor: function (i, node) { return borderDark; },
          }
        },];

        var row = [{ text: '', style: 'th', borderColor: [thFill, thFill, borderLight, borderLight] },
        { text: 'Raw Water', style: 'th', borderColor: [thFill, thFill, borderLight, borderLight] },
        { text: 'Feed Water', style: 'th', borderColor: [thFill, thFill, borderLight, borderLight] }];

        if (!projection.enablePass2) {
          section[1].table.widths.push('*');
          row.push({ text: 'Concentrate', style: 'th', borderColor: [thFill, thFill, thFill, borderLight] });
        }
        else {
          section[1].table.widths.push('*', '*');
          row.push({ text: 'Pass 1 Concentrate', style: 'th', borderColor: [thFill, thFill, borderLight, borderLight] });
          row.push({ text: 'Pass 2 Concentrate', style: 'th', borderColor: [thFill, thFill, thFill, borderLight] });
        }
        section[1].table.body.push(row);

        row = [{ text: 'Langelier Saturation Index', style: 'thVertAltLight' },
        { text: projection.rawWater.lsi.toFixed(2), style: 'td' }, { text: projection.feedWater.lsi.toFixed(2), style: 'td' },
        { text: projection.pass1Conc.lsi.toFixed(2), style: 'td' }];
        if (projection.enablePass2)
          row.push({ text: projection.pass2Conc.lsi.toFixed(2), style: 'td' });
        section[1].table.body.push(row);

        row = [{ text: 'S and DSI Index', style: 'thVert' },
        { text: projection.rawWater.sdsi.toFixed(2), style: 'tdAlt' }, { text: projection.feedWater.sdsi.toFixed(2), style: 'tdAlt' },
        { text: projection.pass1Conc.sdsi.toFixed(2), style: 'tdAlt' }];
        if (projection.enablePass2)
          row.push({ text: projection.pass2Conc.sdsi.toFixed(2), style: 'tdAlt' });
        section[1].table.body.push(row);

        row = [{ text: 'CaSO4/Ksp*100%', style: 'thVertAltLight' },
        { text: projection.rawWater.kspCaSO4.toFixed(2), style: 'td' }, { text: projection.feedWater.kspCaSO4.toFixed(2), style: 'td' },
        { text: projection.pass1Conc.kspCaSO4.toFixed(2), style: 'td' }];
        if (projection.enablePass2)
          row.push({ text: projection.pass2Conc.kspCaSO4.toFixed(2), style: 'td' });
        section[1].table.body.push(row);

        row = [{ text: 'SrSO4/Ksp*100%', style: 'thVert' },
        { text: projection.rawWater.kspSrSO4.toFixed(2), style: 'tdAlt' }, { text: projection.feedWater.kspSrSO4.toFixed(2), style: 'tdAlt' },
        { text: projection.pass1Conc.kspSrSO4.toFixed(2), style: 'tdAlt' }];
        if (projection.enablePass2)
          row.push({ text: projection.pass2Conc.kspSrSO4.toFixed(2), style: 'tdAlt' });
        section[1].table.body.push(row);

        row = [{ text: 'BaSO4/Ksp*100%', style: 'thVertAltLight' },
        { text: projection.rawWater.kspBaSO4.toFixed(2), style: 'td' }, { text: projection.feedWater.kspBaSO4.toFixed(2), style: 'td' },
        { text: projection.pass1Conc.kspBaSO4.toFixed(2), style: 'td' }];
        if (projection.enablePass2)
          row.push({ text: projection.pass2Conc.kspBaSO4.toFixed(2), style: 'td' });
        section[1].table.body.push(row);

        row = [{ text: 'CaF2/Ksp*100%', style: 'thVert' },
        { text: projection.rawWater.kspCaF2.toFixed(2), style: 'tdAlt' }, { text: projection.feedWater.kspCaF2.toFixed(2), style: 'tdAlt' },
        { text: projection.pass1Conc.kspCaF2.toFixed(2), style: 'tdAlt' }];
        if (projection.enablePass2)
          row.push({ text: projection.pass2Conc.kspCaF2.toFixed(2), style: 'tdAlt' });
        section[1].table.body.push(row);

        row = [{ text: 'SiO2 Saturation', style: 'thVertAltLight' },
        { text: projection.rawWater.silicaSat.toFixed(2), style: 'td' }, { text: projection.feedWater.silicaSat.toFixed(2), style: 'td' },
        { text: projection.pass1Conc.silicaSat.toFixed(2), style: 'td' }];
        if (projection.enablePass2)
          row.push({ text: projection.pass2Conc.silicaSat.toFixed(2), style: 'td' });
        section[1].table.body.push(row);

        // row = [{ text: 'Fe (ppm)', style: 'thVert' },
        // { text: projection.rawWater.fe.toFixed(2), style: 'tdAlt' }, { text: projection.feedWater.fe.toFixed(2), style: 'tdAlt' },
        // { text: projection.pass1Conc.fe.toFixed(2), style: 'tdAlt' }];
        // if (projection.enablePass2)
        //   row.push({ text: projection.pass2Conc.fe.toFixed(2), style: 'tdAlt' });
        // section[1].table.body.push(row);

        // row = [{ text: 'Al (ppm)', style: 'thVertAltLight' },
        // { text: projection.rawWater.al.toFixed(2), style: 'td' }, { text: projection.feedWater.al.toFixed(2), style: 'td' },
        // { text: projection.pass1Conc.al.toFixed(2), style: 'td' }];
        // if (projection.enablePass2)
        //   row.push({ text: projection.pass2Conc.al.toFixed(2), style: 'td' });
        // section[1].table.body.push(row);

        var adj =
          row = [{ text: `Osmotic Pressure, ${pressureUnit}`, style: 'thVert' },
          { text: (projection.rawWater.osmoticPressure / unitAdj).toFixed(2), style: 'tdAlt' },
          { text: (projection.feedWater.osmoticPressure / unitAdj).toFixed(2), style: 'tdAlt' },
          { text: (projection.pass1Conc.osmoticPressure / unitAdj).toFixed(2), style: 'tdAlt' }];
        if (projection.enablePass2)
          row.push({ text: (projection.pass2Conc.osmoticPressure / unitAdj).toFixed(2), style: 'tdAlt' });
        section[1].table.body.push(row);

        return section;
      }

      function docSolubilityDesignWarning(projection) {
        var sectionLabel = projection.enablePass2 ? ' - 1st Pass' : '';
        var section = [];

        if (projection.pass[0].warnings.trim() != "") {
          var warnings = [{ text: 'Solubility & Design Warning' + sectionLabel, style: 'section', margin: [0, 30, 0, 0] },
          { text: projection.pass[0].warnings }];
          section.push(warnings);
        }

        if (projection.enablePass2 && projection.pass[1].warnings.trim() != "") {
          var warnings = [{ text: 'Solubility & Design Warning - 2nd Pass', style: 'section', margin: [0, 30, 0, 0] },
          { text: projection.pass[1].warnings }];
          section.push(warnings);
        }
        section.push([{ text: '*** Calculations are accurate within +/- 10% ***', margin: [0, 15, 0, 0] }]);
        section.push([{ text: '*** This projection is not to be used for warranty purposes ***', margin: [0, 15, 0, 0] }]);
        return section;
      }

      function docLastPageFooter() {
        return function (pageNum, pageCount) {
          if (pageNum == pageCount) {
            return [
              {
                table: {
                  widths: ['*', '*', '126'],
                  body: [
                    [{ text: 'Contact', fontSize: 12, bold: true, colSpan: 3 }, {}, {}],
                    [{ text: 'Europe', bold: true }, { text: 'Americas', bold: true }, { text: 'Asia', bold: true }],
                    [{ text: 'Germany: +49 611 962 6001' }, { text: 'USA: +1 805 964 8003' }, { text: 'China: +86 592 677 5500' }],
                    [{ text: 'Italy: +39 0721 1796201 ' }, { text: 'Brazil: +55 11 3378 7500' }, { text: 'Singapore: +65 6457 7533' }],
                    [{ text: 'info@microdyn-nadir.com' }, { text: 'info@microdyn-nadir.com' }, { text: 'infochina@microdyn-nadir.cn' }],
                  ]
                },
                layout: 'noBorders',
                margin: [35, -150, 35, 0]
              },
              { svg: logoSvg, width: 200, alignment: 'right', margin: [0, 20, 18, 0] }]
          }
        };
      }

      var options = {
        // ...
      }

      // For testing output to file - remove unless needed for printing, if so, put in a better location
      console.log(debug);
      if (debug) {
        var pdfDocFile = printer.createPdfKitDocument(docDefinition, options);
        var writeStream = fs.createWriteStream('test.pdf')
        pdfDocFile.pipe(writeStream);
        pdfDocFile.end();
      }

      console.log("report.createReport() start base64");
      var pdfDoc = printer.createPdfKitDocument(docDefinition, options);
      getDoc(pdfDoc, function (err, buffer, pages) {
        var base64 = buffer.toString('base64');
        console.log("report.createReport() end base64");
        //console.log(base64);
        resolve(base64);
      });

    })
  }
};

async function getDoc(pdfDoc, cb) {
  // buffer the output
  var chunks = [];

  pdfDoc.on('data', function (chunk) {
    chunks.push(chunk);
  });
  pdfDoc.on('end', function () {
    var result = Buffer.concat(chunks);
    cb(null, result, pdfDoc._pdfMakePages);
  });

  // close the stream
  pdfDoc.end();
}

var ions = [
  { name: "Calcium", abbr: "Ca", meqL: 20.04, CaCO3: 0.4 },
  { name: "Magnesium", abbr: "Mg", meqL: 12.16, CaCO3: 0.24 },
  { name: "Sodium", abbr: "Na", meqL: 22.99, CaCO3: 0.46 },
  { name: "Potassium", abbr: "K", meqL: 39.1, CaCO3: 0.78 },
  { name: "Ammonium", abbr: "NH4", meqL: 18.04, CaCO3: 0.36 },
  { name: "Barium", abbr: "Ba", meqL: 68.67, CaCO3: 1.37 },
  { name: "Strontium", abbr: "Sr", meqL: 43.81, CaCO3: 0.88 },
  { name: "Carbonate", abbr: "CO3", meqL: 30, CaCO3: 0.6 },
  { name: "Bicarbonate", abbr: "HCO3", meqL: 61.01, CaCO3: 1.22 },
  { name: "Sulfate", abbr: "SO4", meqL: 48.03, CaCO3: 0.96 },
  { name: "Chloride", abbr: "Cl", meqL: 35.45, CaCO3: 0.71 },
  { name: "Fluoride", abbr: "F", meqL: 19, CaCO3: 0.38 },
  { name: "Nitrate", abbr: "NO3", meqL: 62, CaCO3: 1.24 },
  { name: "Silica", abbr: "SiO2", meqL: 60.08, CaCO3: 1.2 }
];

function TDSmeqLTotal(arr) {
  var total = 0;
  arr.forEach((val) => {
    total += val;
  });
  return total;
}

function TDSCaCO3Total(arr) {
  var total = 0;
  arr.forEach(function (value, index) {
    total += (arr[index] / ions[index].CaCO3);
  });
  return total;
}

let logoSvg = `<svg xmlns="http://www.w3.org/2000/svg" width="539" height="83.156" viewBox="0 0 539 83.156">
<g id="MIC_NAD_Logo_RGB" transform="translate(-84.337 -81.325)">
  <g id="Logo" transform="translate(84.337 81.325)">
    <g id="Group_1" data-name="Group 1" transform="translate(86.024 61.25)">
      <path id="Path_2" data-name="Path 2" d="M348.353,264.148h1.853l7.986,17.648h-2.128l-2.054-4.632h-9.535L342.4,281.8h-2.029Zm4.88,11.215-3.98-8.912-4,8.912Z" transform="translate(-340.367 -263.972)" fill="gray"/>
      <path id="Path_3" data-name="Path 3" d="M426.1,264.521h2l6.384,9.562,6.384-9.562h2v17.523H440.9V267.851l-6.384,9.387h-.1l-6.383-9.362v14.168H426.1Z" transform="translate(-397.296 -264.219)" fill="gray"/>
      <path id="Path_4" data-name="Path 4" d="M494.326,264.148h1.849l7.99,17.648h-2.132l-2.05-4.632h-9.538l-2.079,4.632h-2.029Zm4.88,11.215-3.98-8.912-4.005,8.912Z" transform="translate(-437.292 -263.972)" fill="gray"/>
      <path id="Path_5" data-name="Path 5" d="M549.711,264.521h1.853l11.042,14.043V264.521h1.927v17.523h-1.578l-11.317-14.37v14.37h-1.927Z" transform="translate(-479.373 -264.219)" fill="gray"/>
      <path id="Path_6" data-name="Path 6" d="M608.611,264.521h1.849L621.5,278.564V264.521h1.93v17.523h-1.581l-11.313-14.37v14.37h-1.927Z" transform="translate(-518.483 -264.219)" fill="gray"/>
      <path id="Path_7" data-name="Path 7" d="M670.381,279.378H665.2v-1.8h5.18v-5.156h1.878v5.156h5.18v1.8h-5.18v5.156h-1.878Z" transform="translate(-556.06 -269.464)" fill="gray"/>
      <path id="Path_8" data-name="Path 8" d="M714.1,264.521h1.976v7.785h10.114v-7.785h1.976v17.523h-1.976v-7.885H716.078v7.885H714.1Z" transform="translate(-588.53 -264.219)" fill="gray"/>
      <path id="Path_9" data-name="Path 9" d="M769.946,274.709V264.521h1.976v10.063c0,3.78,2.029,5.907,5.357,5.907,3.229,0,5.283-1.951,5.283-5.782V264.521h1.976v10.038c0,5.106-2.929,7.76-7.308,7.76C772.9,282.319,769.946,279.665,769.946,274.709Z" transform="translate(-625.611 -264.219)" fill="gray"/>
      <path id="Path_10" data-name="Path 10" d="M827.345,264.521h2l6.38,9.562,6.387-9.562h2v17.523h-1.98V267.851l-6.38,9.387h-.1l-6.388-9.362v14.168h-1.927Z" transform="translate(-663.725 -264.219)" fill="gray"/>
      <path id="Path_11" data-name="Path 11" d="M892.064,264.521h2l6.38,9.562,6.384-9.562h2v17.523h-1.976V267.851l-6.38,9.387h-.1l-6.384-9.362v14.168h-1.927Z" transform="translate(-706.699 -264.219)" fill="gray"/>
      <path id="Path_12" data-name="Path 12" d="M956.783,264.521h12.661v1.8H958.759v5.983h9.563v1.8h-9.563v6.131h10.813v1.8H956.783Z" transform="translate(-749.673 -264.219)" fill="gray"/>
      <path id="Path_13" data-name="Path 13" d="M1006.735,264.521h1.976v15.695h9.86v1.828h-11.836Z" transform="translate(-782.841 -264.219)" fill="gray"/>
      <path id="Path_14" data-name="Path 14" d="M1072.924,272.736v-.051a8.829,8.829,0,0,1,8.865-9.062,9.119,9.119,0,0,1,6.832,2.779l-1.352,1.452a7.624,7.624,0,0,0-5.509-2.4,6.871,6.871,0,0,0-6.782,7.185v.049a6.915,6.915,0,0,0,6.782,7.236,7.611,7.611,0,0,0,5.66-2.529l1.3,1.276a9.122,9.122,0,0,1-7.008,3.079A8.742,8.742,0,0,1,1072.924,272.736Z" transform="translate(-826.791 -263.623)" fill="gray"/>
      <path id="Path_15" data-name="Path 15" d="M1126.153,284.144v-.049a6.747,6.747,0,0,1,13.495-.051v.051a6.72,6.72,0,0,1-6.783,6.758A6.631,6.631,0,0,1,1126.153,284.144Zm11.518,0v-.049a4.9,4.9,0,0,0-4.807-5.057,4.776,4.776,0,0,0-4.732,5.006v.051a4.853,4.853,0,0,0,4.781,5.03A4.777,4.777,0,0,0,1137.671,284.144Z" transform="translate(-862.136 -272.728)" fill="gray"/>
      <path id="Path_16" data-name="Path 16" d="M1176.714,277.61h1.927v2.177a4.73,4.73,0,0,1,4.178-2.452,4.395,4.395,0,0,1,4.157,2.578,5.156,5.156,0,0,1,4.556-2.578c2.978,0,4.806,2,4.806,5.207v8.01h-1.927v-7.56c0-2.5-1.253-3.9-3.356-3.9-1.952,0-3.554,1.452-3.554,4.005v7.46h-1.927v-7.61c0-2.428-1.277-3.855-3.328-3.855s-3.606,1.7-3.606,4.081v7.384h-1.927Z" transform="translate(-895.709 -272.728)" fill="gray"/>
      <path id="Path_17" data-name="Path 17" d="M1247.609,277.61h1.927v2.6a5.82,5.82,0,0,1,4.958-2.879c3.1,0,6.183,2.452,6.183,6.708v.051c0,4.229-3.052,6.733-6.183,6.733a5.842,5.842,0,0,1-4.958-2.754v6.484h-1.927Zm11.088,6.509v-.051c0-3.052-2.1-5.006-4.552-5.006a4.824,4.824,0,0,0-4.683,4.981v.051a4.815,4.815,0,0,0,4.683,5.006C1256.646,289.1,1258.7,287.247,1258.7,284.119Z" transform="translate(-942.784 -272.728)" fill="gray"/>
      <path id="Path_18" data-name="Path 18" d="M1294.2,287v-.049c0-2.73,2.255-4.182,5.533-4.182a13.759,13.759,0,0,1,3.98.551v-.45c0-2.327-1.426-3.531-3.854-3.531a9.012,9.012,0,0,0-3.931.953l-.579-1.577a10.7,10.7,0,0,1,4.708-1.076,5.679,5.679,0,0,1,4.182,1.426,5.1,5.1,0,0,1,1.327,3.781v7.91h-1.853V288.8a5.672,5.672,0,0,1-4.679,2.229C1296.627,291.026,1294.2,289.65,1294.2,287Zm9.539-1v-1.251a13.49,13.49,0,0,0-3.8-.55c-2.428,0-3.78,1.052-3.78,2.678v.049c0,1.629,1.5,2.58,3.254,2.58C1301.786,289.5,1303.738,288.048,1303.738,285.994Z" transform="translate(-973.72 -272.926)" fill="gray"/>
      <path id="Path_19" data-name="Path 19" d="M1340.276,277.61h1.93v2.253a4.893,4.893,0,0,1,4.429-2.528c3.13,0,4.958,2.1,4.958,5.182v8.035h-1.931v-7.56c0-2.4-1.3-3.9-3.578-3.9a3.807,3.807,0,0,0-3.878,4.055v7.411h-1.93Z" transform="translate(-1004.316 -272.728)" fill="gray"/>
      <path id="Path_20" data-name="Path 20" d="M1392.195,278.154h2.054l-5.434,13.367c-1.1,2.679-2.35,3.656-4.305,3.656a5.887,5.887,0,0,1-2.753-.651l.65-1.528a4.052,4.052,0,0,0,2.032.476c1.15,0,1.878-.6,2.65-2.428l-5.882-12.893h2.128l4.708,10.79Z" transform="translate(-1031.494 -273.272)" fill="gray"/>
    </g>
    <g id="Group_14" data-name="Group 14" transform="translate(0 0.585)">
      <path id="Path_21" data-name="Path 21" d="M161.141,169.2V122.089a27.265,27.265,0,0,0-3.169-1.643v50.4A27.1,27.1,0,0,0,161.141,169.2Zm-10,3.641V118.45a27.269,27.269,0,0,0-3.4-.3v55A28.13,28.13,0,0,0,151.141,172.843Zm4.943-1.25V119.7a27.473,27.473,0,0,0-3.284-.933v53.765A27.238,27.238,0,0,0,156.085,171.593Zm14.809-12.479V132.177a27.5,27.5,0,0,0-2.443-3.635v34.207A27.591,27.591,0,0,0,170.894,159.114Zm-51.48-13.468a27.56,27.56,0,0,0,.905,7.015V138.629A27.583,27.583,0,0,0,119.414,145.646ZM165.837,165.6V125.686a27.625,27.625,0,0,0-2.806-2.333v44.585A27.494,27.494,0,0,0,165.837,165.6ZM137.742,119.7v51.894a27.624,27.624,0,0,0,3.284.936V118.764A27.247,27.247,0,0,0,137.742,119.7Zm-14.809,12.48v26.932a27.5,27.5,0,0,0,2.443,3.636v-34.2A27.655,27.655,0,0,0,122.933,132.179Zm5.057-6.491V165.6a27.7,27.7,0,0,0,2.807,2.336V123.354A27.634,27.634,0,0,0,127.99,125.688Zm46.424,19.959a27.572,27.572,0,0,0-.906-7.022v14.041A27.541,27.541,0,0,0,174.413,145.646Zm-41.728-23.555V169.2a27.543,27.543,0,0,0,3.169,1.642v-50.4A26.972,26.972,0,0,0,132.685,122.091Zm10-3.641v54.393a28.154,28.154,0,0,0,3.4.3v-55A27.292,27.292,0,0,0,142.684,118.45Z" transform="translate(-107.628 -106.36)" fill="#46af28"/>
      <g id="Group_13" data-name="Group 13">
        <rect id="Rectangle_2" data-name="Rectangle 2" width="78.57" height="6.079" transform="translate(0 72.492)" fill="#00823c"/>
        <rect id="Rectangle_3" data-name="Rectangle 3" width="78.57" height="6.08" fill="#00823c"/>
        <path id="Path_22" data-name="Path 22" d="M84.337,105.9v3.669H118.1a28.522,28.522,0,0,1,11.04,0h33.765V105.9Z" transform="translate(-84.337 -98.231)" fill="#00823c"/>
        <g id="Group_2" data-name="Group 2" transform="translate(0 23.472)">
          <path id="Path_23" data-name="Path 23" d="M99.932,152.923H84.337v1.852H98.8C99.154,154.143,99.53,153.524,99.932,152.923Z" transform="translate(-84.337 -152.923)" fill="#00823c"/>
          <path id="Path_24" data-name="Path 24" d="M271.769,152.923c.4.6.777,1.22,1.134,1.852h14.461v-1.852Z" transform="translate(-208.794 -152.923)" fill="#00823c"/>
        </g>
        <g id="Group_3" data-name="Group 3" transform="translate(0 28.435)">
          <path id="Path_25" data-name="Path 25" d="M97.284,167.7H84.337v1.852H96.6C96.808,168.92,97.034,168.3,97.284,167.7Z" transform="translate(-84.337 -167.695)" fill="#00823c"/>
          <path id="Path_26" data-name="Path 26" d="M279.647,167.7c.25.607.477,1.225.686,1.852h12.262V167.7Z" transform="translate(-214.025 -167.695)" fill="#00823c"/>
        </g>
        <g id="Group_4" data-name="Group 4" transform="translate(0 33.508)">
          <path id="Path_27" data-name="Path 27" d="M95.727,182.795H84.337v1.63H95.445C95.525,183.877,95.617,183.332,95.727,182.795Z" transform="translate(-84.337 -182.795)" fill="#00823c"/>
          <path id="Path_28" data-name="Path 28" d="M284.282,182.795c.111.537.2,1.082.283,1.63h11.108v-1.63Z" transform="translate(-217.102 -182.795)" fill="#00823c"/>
        </g>
        <g id="Group_5" data-name="Group 5" transform="translate(0 38.472)">
          <path id="Path_29" data-name="Path 29" d="M95.141,198.376c0-.271.013-.539.021-.808H84.337v1.63H95.162C95.155,198.924,95.141,198.652,95.141,198.376Z" transform="translate(-84.337 -197.568)" fill="#00823c"/>
          <path id="Path_30" data-name="Path 30" d="M285.964,197.568c.007.269.021.537.021.808s-.013.548-.021.822h10.825v-1.63Z" transform="translate(-218.219 -197.568)" fill="#00823c"/>
        </g>
        <g id="Group_6" data-name="Group 6" transform="translate(0 48.285)">
          <path id="Path_31" data-name="Path 31" d="M96.6,226.775H84.337v1.854H97.29C97.039,228.022,96.812,227.4,96.6,226.775Z" transform="translate(-84.337 -226.775)" fill="#00823c"/>
          <path id="Path_32" data-name="Path 32" d="M280.317,226.775c-.209.628-.436,1.247-.687,1.854h12.953v-1.854Z" transform="translate(-214.013 -226.775)" fill="#00823c"/>
        </g>
        <g id="Group_7" data-name="Group 7" transform="translate(0 43.379)">
          <path id="Path_33" data-name="Path 33" d="M95.727,213.8H84.337v-1.63H95.445C95.525,212.72,95.617,213.264,95.727,213.8Z" transform="translate(-84.337 -212.172)" fill="#00823c"/>
          <path id="Path_34" data-name="Path 34" d="M284.282,213.8c.111-.538.2-1.082.283-1.63h11.108v1.63Z" transform="translate(-217.102 -212.172)" fill="#00823c"/>
        </g>
        <g id="Group_8" data-name="Group 8" transform="translate(0 53.249)">
          <path id="Path_35" data-name="Path 35" d="M98.805,241.548H84.337V243.4h15.6C99.538,242.8,99.162,242.18,98.805,241.548Z" transform="translate(-84.337 -241.548)" fill="#00823c"/>
          <path id="Path_36" data-name="Path 36" d="M272.879,241.548c-.356.632-.732,1.252-1.135,1.852h15.6v-1.852Z" transform="translate(-208.777 -241.548)" fill="#00823c"/>
        </g>
        <g id="Group_9" data-name="Group 9" transform="translate(0 57.856)">
          <path id="Path_37" data-name="Path 37" d="M260.551,255.261a28.611,28.611,0,0,1-2.5,2.563h20.2v-2.563Z" transform="translate(-199.688 -255.261)" fill="#00823c"/>
          <path id="Path_38" data-name="Path 38" d="M102.044,255.261H84.337v2.563h20.2A28.653,28.653,0,0,1,102.044,255.261Z" transform="translate(-84.337 -255.261)" fill="#00823c"/>
        </g>
        <g id="Group_10" data-name="Group 10" transform="translate(0 62.817)">
          <path id="Path_39" data-name="Path 39" d="M107.587,270.025H84.337v2.566h27.891A28.442,28.442,0,0,1,107.587,270.025Z" transform="translate(-84.337 -270.025)" fill="#00823c"/>
          <path id="Path_40" data-name="Path 40" d="M239.813,270.025a28.432,28.432,0,0,1-4.641,2.566h27.891v-2.566Z" transform="translate(-184.493 -270.025)" fill="#00823c"/>
        </g>
        <path id="Path_41" data-name="Path 41" d="M129.105,283.152a28.478,28.478,0,0,1-10.967,0h-33.8v3.669h78.57v-3.669Z" transform="translate(-84.337 -215.925)" fill="#00823c"/>
        <g id="Group_11" data-name="Group 11" transform="translate(0 13.189)">
          <path id="Path_42" data-name="Path 42" d="M107.587,124.885H84.337v-2.566h27.891A28.443,28.443,0,0,0,107.587,124.885Z" transform="translate(-84.337 -122.319)" fill="#00823c"/>
          <path id="Path_43" data-name="Path 43" d="M239.813,124.885a28.433,28.433,0,0,0-4.641-2.566h27.891v2.566Z" transform="translate(-184.493 -122.319)" fill="#00823c"/>
        </g>
        <g id="Group_12" data-name="Group 12" transform="translate(0 18.153)">
          <path id="Path_44" data-name="Path 44" d="M260.551,139.655a28.615,28.615,0,0,0-2.5-2.563h20.2v2.563Z" transform="translate(-199.688 -137.092)" fill="#00823c"/>
          <path id="Path_45" data-name="Path 45" d="M102.044,139.655H84.337v-2.563h20.2A28.657,28.657,0,0,0,102.044,139.655Z" transform="translate(-84.337 -137.092)" fill="#00823c"/>
        </g>
      </g>
    </g>
    <g id="Group_15" data-name="Group 15" transform="translate(86.024 0)">
      <path id="Path_46" data-name="Path 46" d="M340.367,83.139H348.8l9.34,15.028,9.344-15.028h8.428v35.538h-7.768v-23.2l-10,15.179h-.2l-9.9-15.028v23.05h-7.667Z" transform="translate(-340.367 -82.53)" fill="#00823c"/>
      <path id="Path_47" data-name="Path 47" d="M469.631,83.139h7.819v35.538h-7.819Z" transform="translate(-426.199 -82.53)" fill="#00823c"/>
      <path id="Path_48" data-name="Path 48" d="M512.688,99.806v-.1a18.062,18.062,0,0,1,18.529-18.378c6.7,0,10.714,2.234,14.014,5.483l-4.977,5.736c-2.741-2.487-5.532-4.01-9.088-4.01-5.99,0-10.307,4.975-10.307,11.068v.1c0,6.093,4.216,11.17,10.307,11.17,4.064,0,6.549-1.625,9.344-4.164l4.973,5.026a18.164,18.164,0,0,1-14.57,6.347A17.939,17.939,0,0,1,512.688,99.806Z" transform="translate(-454.79 -81.325)" fill="#00823c"/>
      <path id="Path_49" data-name="Path 49" d="M631.928,83.139h16.246c4.519,0,8.021,1.269,10.357,3.6a11.152,11.152,0,0,1,3.044,8.123v.1a11.034,11.034,0,0,1-7.613,11.017l8.68,12.692H653.5L645.888,107.3h-6.142v11.373h-7.818ZM647.666,100.4c3.808,0,5.99-2.03,5.99-5.027v-.1c0-3.35-2.333-5.077-6.142-5.077h-7.768V100.4Z" transform="translate(-533.966 -82.53)" fill="#00823c"/>
      <path id="Path_50" data-name="Path 50" d="M737.533,99.806v-.1c0-10.1,7.973-18.378,18.94-18.378S775.309,89.5,775.309,99.6v.1c0,10.1-7.973,18.379-18.94,18.379S737.533,109.909,737.533,99.806Zm29.6,0v-.1c0-6.091-4.468-11.169-10.765-11.169s-10.66,4.975-10.66,11.068v.1c0,6.093,4.468,11.17,10.765,11.17S767.134,105.9,767.134,99.806Z" transform="translate(-604.089 -81.325)" fill="#00823c"/>
      <path id="Path_51" data-name="Path 51" d="M868.709,83.139h13.863c11.169,0,18.886,7.665,18.886,17.667v.1c0,10-7.718,17.77-18.886,17.77H868.709Zm7.818,7.056V111.62h6.044c6.394,0,10.711-4.315,10.711-10.61v-.1c0-6.3-4.316-10.713-10.711-10.713Z" transform="translate(-691.191 -82.53)" fill="#00823c"/>
      <path id="Path_52" data-name="Path 52" d="M979.876,104.665,966.219,83.139h9.138l8.478,14.267,8.634-14.267h8.883l-13.657,21.373v14.165h-7.818Z" transform="translate(-755.938 -82.53)" fill="#00823c"/>
      <path id="Path_53" data-name="Path 53" d="M1085.335,83.139h7.209L1109.2,105.02V83.139h7.714v35.538h-6.65l-17.209-22.593v22.593h-7.717Z" transform="translate(-835.032 -82.53)" fill="#00823c"/>
      <path id="Path_54" data-name="Path 54" d="M1208.016,83.139h3.758l22.392,28.481V83.139h3.909v35.538h-3.2l-22.947-29.143v29.143h-3.913Z" transform="translate(-916.494 -82.53)" fill="#00823c"/>
      <path id="Path_55" data-name="Path 55" d="M1330.57,82.382h3.758l16.193,35.793h-4.313l-4.165-9.393H1322.7l-4.216,9.393h-4.111Zm9.9,22.745L1332.4,87.052l-8.125,18.075Z" transform="translate(-987.117 -82.027)" fill="#00823c"/>
      <path id="Path_56" data-name="Path 56" d="M1439.86,83.139H1452.2c11.169,0,18.886,7.665,18.886,17.667v.1c0,10-7.718,17.77-18.886,17.77H1439.86Zm4.01,3.706v28.127h8.327c8.987,0,14.725-6.093,14.725-13.962v-.1c0-7.869-5.738-14.063-14.725-14.063Z" transform="translate(-1070.44 -82.53)" fill="#00823c"/>
      <path id="Path_57" data-name="Path 57" d="M1559.136,83.139h4.01v35.538h-4.01Z" transform="translate(-1149.641 -82.53)" fill="#00823c"/>
      <path id="Path_58" data-name="Path 58" d="M1601.059,83.139h15.283c4.364,0,7.869,1.32,10.1,3.554a9.811,9.811,0,0,1,2.745,7.005v.1c0,5.891-4.064,9.342-9.65,10.357l10.916,14.521h-4.923l-10.307-13.809h-10.155v13.809h-4.01Zm14.927,18.124c5.33,0,9.139-2.741,9.139-7.31v-.1c0-4.367-3.351-7.007-9.088-7.007h-10.967v14.418Z" transform="translate(-1177.478 -82.53)" fill="#00823c"/>
    </g>
  </g>
</g>
</svg>
`;

